% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ancom.R
\name{ancom}
\alias{ancom}
\title{Analysis of Composition of Microbiomes (ANCOM)}
\usage{
ancom(
  data = NULL,
  taxa_are_rows = TRUE,
  assay.type = NULL,
  assay_name = "counts",
  rank = NULL,
  tax_level = NULL,
  aggregate_data = NULL,
  meta_data = NULL,
  p_adj_method = "holm",
  prv_cut = 0.1,
  lib_cut = 0,
  main_var,
  adj_formula = NULL,
  rand_formula = NULL,
  lme_control = lme4::lmerControl(),
  struc_zero = FALSE,
  neg_lb = FALSE,
  alpha = 0.05,
  n_cl = 1,
  verbose = TRUE
)
}
\arguments{
\item{data}{the input data. The \code{data} parameter should be either a
\code{matrix}, \code{data.frame}, \code{phyloseq} or a \code{TreeSummarizedExperiment}
object. Both \code{phyloseq} and \code{TreeSummarizedExperiment} objects
consist of a feature table (microbial count table), a sample metadata table,
a taxonomy table (optional), and a phylogenetic tree (optional).
If a \code{matrix} or \code{data.frame} is provided, ensure that the row
names of the \code{metadata} match the sample names (column names if
\code{taxa_are_rows} is TRUE, and row names otherwise) in \code{data}.
if a \code{phyloseq} or a \code{TreeSummarizedExperiment} is used, this
standard has already been enforced. For detailed information, refer to
\code{?phyloseq::phyloseq} or
\code{?TreeSummarizedExperiment::TreeSummarizedExperiment}.
It is recommended to use low taxonomic levels, such as OTU or species level,
as the estimation of sampling fractions requires a large number of taxa.}

\item{taxa_are_rows}{logical. Whether taxa are positioned in the rows of the
feature table. Default is TRUE.
It is recommended to use low taxonomic levels, such as OTU or species level,
as the estimation of sampling fractions requires a large number of taxa.}

\item{assay.type}{alias for \code{assay_name}.}

\item{assay_name}{character. Name of the count table in the data object
(only applicable if data object is a \code{(Tree)SummarizedExperiment}).
Default is "counts".
See \code{?SummarizedExperiment::assay} for more details.}

\item{rank}{alias for \code{tax_level}.}

\item{tax_level}{character. The taxonomic or non taxonomic(rowData) level of interest. The input data
can be analyzed at any taxonomic or rowData level without prior agglomeration.
Note that \code{tax_level} must be a value from \code{taxonomyRanks} or \code{rowData}, which
includes "Kingdom", "Phylum" "Class", "Order", "Family" "Genus" "Species" etc.
See \code{?mia::taxonomyRanks} for more details.
Default is NULL, i.e., do not perform agglomeration, and the
ANCOM-BC2 analysis will be performed at the lowest taxonomic level of the
input \code{data}.}

\item{aggregate_data}{The abundance data that has been aggregated to the desired
taxonomic level. This parameter is required only when the input data is in
\code{matrix} or \code{data.frame} format. For \code{phyloseq} or \code{TreeSummarizedExperiment}
data, aggregation is performed by specifying the \code{tax_level} parameter.}

\item{meta_data}{a \code{data.frame} containing sample metadata.
This parameter is mandatory when the input \code{data} is a generic
\code{data.frame}. Ensure that the row names of the \code{metadata} match the
sample names (column names if \code{taxa_are_rows} is TRUE, and row names
otherwise) in \code{data}.}

\item{p_adj_method}{character. method to adjust p-values. Default is "holm".
Options include "holm", "hochberg", "hommel", "bonferroni", "BH", "BY",
"fdr", "none". See \code{?stats::p.adjust} for more details.}

\item{prv_cut}{a numerical fraction between 0 and 1. Taxa with prevalences
(the proportion of samples in which the taxon is present)
less than \code{prv_cut} will be excluded in the analysis. For example,
if there are 100 samples, and a taxon has nonzero counts present in less than
100*prv_cut samples, it will not be considered in the analysis.
Default is 0.10.}

\item{lib_cut}{a numerical threshold for filtering samples based on library
sizes. Samples with library sizes less than \code{lib_cut} will be
excluded in the analysis. Default is 0, i.e. do not discard any sample.}

\item{main_var}{character. The name of the main variable of interest.}

\item{adj_formula}{character string representing the formula for
covariate adjustment. Please note that you should NOT include the
\code{main_var} in the formula. Default is \code{NULL}.}

\item{rand_formula}{the character string expresses how the microbial absolute
abundances for each taxon depend on the random effects in metadata. ANCOM
follows the \code{lmerTest} package in formulating the random effects. See
\code{?lmerTest::lmer} for more details. Default is \code{NULL}.}

\item{lme_control}{a list of control parameters for mixed model fitting.
See \code{?lme4::lmerControl} for details.}

\item{struc_zero}{logical. whether to detect structural zeros based on
\code{main_var}. \code{main_var} should be discrete. Default is FALSE.}

\item{neg_lb}{logical. whether to classify a taxon as a structural zero using
its asymptotic lower bound. Default is FALSE.}

\item{alpha}{numeric. level of significance. Default is 0.05.}

\item{n_cl}{numeric. The number of nodes to be forked. For details, see
\code{?parallel::makeCluster}. Default is 1 (no parallel computing).}

\item{verbose}{logical. Whether to display detailed progress messages.}
}
\value{
a \code{list} with components:
        \itemize{
        \item{ \code{res},  a \code{data.frame} containing ANCOM
        result for the variable specified in \code{main_var},
        each column is:}
        \itemize{
        \item{ \code{W}, test statistics.}
        \item{ \code{detected_0.9, detected_0.8, detected_0.7, detected_0.6},
        logical vectors representing whether a taxon is differentially
        abundant under a series of cutoffs. For example, TRUE in
        \code{detected_0.7} means the number of ALR transformed models where
        the taxon is differentially abundant with regard to the main variable
        outnumbers \code{0.7 * (n_tax - 1)}. \code{detected_0.7} is commonly
        used. Choose \code{detected_0.8} or \code{detected_0.9} for more
        conservative results, or choose \code{detected_0.6} for more liberal
        results.}
        }
        \item{ \code{zero_ind}, a logical \code{data.frame} with TRUE
        indicating the taxon is detected to contain structural zeros in
        some specific groups.}
        \item{ \code{beta_data}, a numeric \code{matrix} containing pairwise
        coefficients for the main variable of interest in ALR transformed
        regression models.}
        \item{ \code{p_data}, a numeric \code{matrix} containing pairwise
        p-values for the main variable of interest in ALR transformed
        regression models.}
        \item{ \code{q_data}, a numeric \code{matrix} containing adjusted
        p-values by applying the \code{p_adj_method} to the \code{p_data}
        matrix.}
        }
}
\description{
Determine taxa whose absolute abundances, per unit volume, of
the ecosystem (e.g. gut) are significantly different with changes in the
covariate of interest (e.g. group). The current version of
\code{ancom} function implements ANCOM in cross-sectional and repeated
measurements data while allowing for covariate adjustment.
}
\details{
A taxon is considered to have structural zeros in some (>=1)
groups if it is completely (or nearly completely) missing in these groups.
For instance, suppose there are three groups: g1, g2, and g3.
If the counts of taxon A in g1 are 0 but nonzero in g2 and g3,
then taxon A will be considered to contain structural zeros in g1.
In this example, taxon A is declared to be differentially abundant between
g1 and g2, g1 and g3, and consequently, it is globally differentially
abundant with respect to this group variable.
Such taxa are not further analyzed using ANCOM, but the results are
summarized in the overall summary. For more details about the structural
zeros, please go to the
\href{https://doi.org/10.3389/fmicb.2017.02114}{ANCOM-II} paper.
Setting \code{neg_lb = TRUE} indicates that you are using both criteria
stated in section 3.2 of
\href{https://doi.org/10.3389/fmicb.2017.02114}{ANCOM-II}
to detect structural zeros; otherwise, the algorithm will only use the
equation 1 in section 3.2 for declaring structural zeros. Generally, it is
recommended to set \code{neg_lb = TRUE} when the sample size per group is
relatively large (e.g. > 30).
}
\examples{
library(ANCOMBC)
if (requireNamespace("microbiome", quietly = TRUE)) {
    data(atlas1006, package = "microbiome")
    # subset to baseline
    pseq = phyloseq::subset_samples(atlas1006, time == 0)

    # run ancom function
    set.seed(123)
    out = ancom(data = pseq, tax_level = "Family",
                p_adj_method = "holm", prv_cut = 0.10, lib_cut = 1000,
                main_var = "bmi_group", adj_formula = "age + nationality",
                rand_formula = NULL, lme_control = NULL,
                struc_zero = TRUE, neg_lb = TRUE, alpha = 0.05, n_cl = 1)

    res = out$res
} else {
    message("The 'microbiome' package is not installed. Please install it to use this example.")
}

}
\references{
\insertRef{mandal2015analysis}{ANCOMBC}

\insertRef{kaul2017analysis}{ANCOMBC}
}
\seealso{
\code{\link{ancombc}} \code{\link{ancombc2}}
}
\author{
Huang Lin
}
