% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/PBMC_10x.R
\name{PBMC_10x}
\alias{PBMC_10x}
\title{10k PBMC data}
\format{
\code{MultiAssayExperiment} obtained from an \code{ArchR} project.
Annotated with the Hg38 genome build.
Contains the following experiments:
\itemize{
\item{\strong{TileMatrix}: SingleCellExperiment with 6062095 rows and 9702 columns}
\item{\strong{GeneScoreMatrix}: SingleCellExperiment with 24919 rows and 9702 columns}
\item{\strong{GeneExpressionMatrix}: SingleCellExperiment with 36438 rows and 9702 columns}
\item{\strong{PeakMatrix}: SingleCellExperiment with 159290 rows and 9702 columns}
\item{\strong{MotifMatrix}: SingleCellExperiment with 870 rows and 9702 columns}
}
}
\usage{
PBMC_10x(
  metadata = FALSE,
  experiments = c("TileMatrix", "GeneScoreMatrix", "GeneExpressionMatrix", "PeakMatrix",
    "MotifMatrix")
)
}
\arguments{
\item{metadata}{logical flag specifying whether to return data or metadata only}

\item{experiments}{character vector of matrices to return; see \code{Format}}
}
\value{
\code{MultiAssayExperiment} made up of \code{SingleCellExperiment}s
with assays stored as \code{DelayedMatrix} objects.
If \code{metadata = TRUE}, an \code{ExperimentHub} object listing this data set's metadata.
}
\description{
PBMC from a Healthy Donor downloaded from 10x Genomics. Granulocytes were removed by cell sorting.
Paired ATAC and Gene Expression libraries were generated from the isolated nuclei.
Targeted nuclei recovery was 10,000.
Data source: https://www.10xgenomics.com/datasets/pbmc-from-a-healthy-donor-granulocytes-removed-through-cell-sorting-10-k-1-standard-2-0-0
}
\section{Data preparation}{

\subsection{1. Download data}{

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# specify output directory
outDir <- <OUTPUT_DIRECTORY>

# dowbnload data from 10XGenomics server    
download.file(c("https://cf.10xgenomics.com/samples/cell-arc/1.0.0/pbmc_granulocyte_sorted_10k/pbmc_granulocyte_sorted_10k_filtered_feature_bc_matrix.h5", "https://cf.10xgenomics.com/samples/cell-arc/1.0.0/pbmc_granulocyte_sorted_10k/pbmc_granulocyte_sorted_10k_atac_fragments.tsv.gz", 
                "https://cf.10xgenomics.com/samples/cell-arc/1.0.0/pbmc_granulocyte_sorted_10k/pbmc_granulocyte_sorted_10k_atac_fragments.tsv.gz.tbi"), c(file.path(outDir,"pbmc_granulocyte_sorted_10k_filtered_feature_bc_matrix.h5"), file.path(outDir,"pbmc_granulocyte_sorted_10k_atac_fragments.tsv.gz"), file.path(outDir,"pbmc_granulocyte_sorted_10k_atac_fragments.tsv.gz.tbi")),
              method="libcurl")
                                                                                                                
}\if{html}{\out{</div>}}
}

\subsection{2. Initiate \code{ArchR} project}{

\if{html}{\out{<div class="sourceCode r">}}\preformatted{library(ArchR)

# configure ArchR
addArchRGenome("hg38")

# create arrow file from fragment files
## list fragment files
fragments <- <FRAGMENT_FILES>
## assign sample names
names(fragments) <- <SAMPLE_IDs>
## create arrows
createArrowFiles(inputFiles = fragments, sampleNames = names(fragments))

# locate arrow files
arrows <- <ARROW_FILES>

# create ArchR project
project <- ArchRProject(arrows, outDir)

}\if{html}{\out{</div>}}
}

\subsection{3. Add gene expression data}{

\if{html}{\out{<div class="sourceCode r">}}\preformatted{seRNA <- import10xFeatureMatrix(
  input = c(file.path(outDir, "pbmc_granulocyte_sorted_10k_filtered_feature_bc_matrix.h5")),
  names = c("PBMC_10k")
)

# filter out genes which are expressed in less than 3 cells
seRNA <- seRNA[colSums(assay(seRNA))>2,]

proj <- addGeneExpressionMatrix(input = proj, seRNA = seRNA, force = TRUE)
}\if{html}{\out{</div>}}
}

\subsection{4. Quality control}{

\if{html}{\out{<div class="sourceCode r">}}\preformatted{#Filter Cells
proj <- proj[proj$TSSEnrichment > 6 & proj$nFrags > 2500 & !is.na(proj$Gex_nUMI)]

#Doublet Filtration
proj <- addDoubletScores(proj)
proj <- filterDoublets(proj)
}\if{html}{\out{</div>}}
}

\subsection{5. Clustering and dimensionality reduction}{

\if{html}{\out{<div class="sourceCode r">}}\preformatted{#LSI-ATAC
proj <- addIterativeLSI(
  ArchRProj = proj,
  clusterParams = list(
    resolution = 0.2,
    sampleCells = 10000,
    n.start = 10
  ),
  saveIterations = FALSE,
  useMatrix = "TileMatrix",
  depthCol = "nFrags",
  name = "LSI_ATAC"
)

#LSI-RNA
proj <- addIterativeLSI(
  ArchRProj = proj,
  clusterParams = list(
    resolution = 0.2,
    sampleCells = 10000,
    n.start = 10
  ),
  saveIterations = FALSE,
  useMatrix = "GeneExpressionMatrix",
  depthCol = "Gex_nUMI",
  varFeatures = 2500,
  firstSelection = "variable",
  binarize = FALSE,
  name = "LSI_RNA"
)

#Combined Dims
proj <- addCombinedDims(proj, reducedDims = c("LSI_ATAC", "LSI_RNA"), name =  "LSI_Combined")

#UMAPs
proj <- addUMAP(proj, reducedDims = "LSI_ATAC", name = "UMAP_ATAC", minDist = 0.8, force = TRUE)

proj <- addUMAP(proj, reducedDims = "LSI_RNA", name = "UMAP_RNA", minDist = 0.8, force = TRUE)

proj <- addUMAP(proj, reducedDims = "LSI_Combined", name = "UMAP_Combined", minDist = 0.8, force = TRUE)

#Add Clusters
proj <- addClusters(proj, reducedDims = "LSI_Combined", name = "Clusters", resolution = 0.4, force = TRUE)
#proj <- addClusters(proj, reducedDims = "LSI_RNA", name = "Clusters_genes", resolution = 0.4, force = TRUE)
}\if{html}{\out{</div>}}
}

\subsection{6. Peak calling}{

\if{html}{\out{<div class="sourceCode r">}}\preformatted{library(BSgenome.Hsapiens.UCSC.hg38)

proj<- addGroupCoverages(ArchRProj = proj, groupBy = "Clusters")

proj <- addReproduciblePeakSet(
  ArchRProj = proj,
  groupBy = "Clusters",
  pathToMacs2 = pathToMacs2,
  force = TRUE,
  threads = 10
)

# add information about sequence motifs recognized by known transcriptions factors
proj <- addMotifAnnotations(ArchRProj = proj,
                               motifSet = "cisbp", name = "Motif")

proj <- addPeakMatrix(proj)

# add background peaks to be compared against during peak variation assessement
proj <- addBgdPeaks(proj)

# calculate per-cell devations of motif annotations
proj <- addDeviationsMatrix(proj, peakAnnotation = "Motif", force = TRUE)
}\if{html}{\out{</div>}}
}

\subsection{7. Cell type annotation}{

\if{html}{\out{<div class="sourceCode r">}}\preformatted{GeneExpressionMatrix <- getMatrixFromProject(proj, useMatrix = "GeneExpressionMatrix")

# Loading reference data with Ensembl annotations.
library(celldex)
#ref.data <- HumanPrimaryCellAtlasData(ensembl=FALSE)
bpe_data <- BlueprintEncodeData()


expr_assay <- assays(GeneExpressionMatrix)[[1]]
rownames(expr_assay) <- rowData(GeneExpressionMatrix)$name

# Performing predictions.
library(SingleR)
predictions <- SingleR(test=expr_assay, assay.type.test=1,
                       ref=bpe_data, labels=bpe_data$label.fine) # could be switched to label.main

proj$cell_type_SingleR <- predictions$labels



library(ArchR)
library(scater)
library(SingleCellExperiment)
library(epiregulon.archr)

GeneExpressionMatrix <- getMatrixFromProject(proj, "GeneExpressionMatrix")
GeneExpressionMatrix <- ArchRMatrix2SCE(GeneExpressionMatrix, rename="normalizedCounts")
reducedDim(GeneExpressionMatrix, "UMAP_Combined") <- getEmbedding(proj, embedding = "UMAP_Combined")
GeneExpressionMatrix$cell_type[is.na(GeneExpressionMatrix$cell_type)] <- "unknown"
plotReducedDim(GeneExpressionMatrix, dimred="UMAP_Combined", colour_by = "Clusters", text_by = "Clusters")
plotReducedDim(GeneExpressionMatrix, dimred="UMAP_Combined", colour_by = "cell_type_SingleR", text_by = "cell_type_SingleR")


# identify cell types in cluster 10
GeneExpressionMatrix$C10 <- GeneExpressionMatrix$cell_type_SingleR
GeneExpressionMatrix$C10[GeneExpressionMatrix$Clusters!="C10"] <- NA
plotReducedDim(GeneExpressionMatrix, dimred="UMAP_Combined", colour_by = "C10", text_by = "C10")

# identify cell types in cluster 11
GeneExpressionMatrix$C11 <- GeneExpressionMatrix$cell_type_SingleR
GeneExpressionMatrix$C11[GeneExpressionMatrix$Clusters!="C11"] <- NA
plotReducedDim(GeneExpressionMatrix, dimred="UMAP_Combined", colour_by = "C11", text_by = "C11")
table(GeneExpressionMatrix$C11)

# identify cell types in cluster 1
GeneExpressionMatrix$C1 <- GeneExpressionMatrix$cell_type_SingleR
GeneExpressionMatrix$C1[GeneExpressionMatrix$Clusters!="C1"] <- NA
table(GeneExpressionMatrix$C1)

clusters <- proj$Clusters
manual_annotation <- rep(NA, length(clusters))
manual_annotation[clusters \%in\% "C6"] <- "Naive CD4+ T"
manual_annotation[clusters \%in\% c("C13")] <- "CD14+ Mono"
manual_annotation[clusters \%in\% "C12"] <- "Monocytes"
manual_annotation[clusters \%in\% c("C2", "C3")] <- "B"
manual_annotation[clusters \%in\% "C4"] <- "Memory CD8+ T"
manual_annotation[clusters \%in\% "C14"] <- "FCGR3A+ Mono"
manual_annotation[clusters \%in\% "C5"] <- "NK"
manual_annotation[clusters \%in\% c("C9", "C10")] <- "Memory CD4+ T"
manual_annotation[clusters \%in\% c("C7", "C8")] <- "Naive CD8+ T"
manual_annotation[clusters \%in\% c("C1")] <- "DC"
proj$cell_type <- manual_annotation
proj <- proj[!is.na(proj$cell_type)]
}\if{html}{\out{</div>}}
}
}

\section{Data storage and access}{

The \code{MultiAssayExperiments} is split into separate \code{SingleCellExperiment}
objects and they in turn are split into components, all of which are stored in a
single hdf5 file. Data and can be accessed with a special function that extracts
elements of the requested experiment(s), reassembles them, and builds an MAE.
}

\examples{
# check metada of dataset
PBMC_10x(metadata = TRUE)
# download data
 \dontrun{
PBMC_10x()
}

}
