## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
    collapse = TRUE,
    comment = "#>"
)

## ----install, eval=FALSE------------------------------------------------------
# if (!require("BiocManager", quietly = TRUE))
#     install.packages("BiocManager")
# 
# BiocManager::install("humanHippocampus2024")
# 
# ## Check that you have a valid Bioconductor installation
# BiocManager::valid()

## ----prep,message=FALSE, warning=FALSE----------------------------------------
library(SummarizedExperiment)
library(SpatialExperiment)
library(humanHippocampus2024)

## ----connect------------------------------------------------------------------
## Connect to ExperimentHub
library(ExperimentHub)
ehub <- ExperimentHub()

## Load the datasets of the package
myfiles <- query(ehub, "humanHippocampus2024")

## Resulting humanHippocampus2024 datasets from ExperimentHub query
myfiles

## ----spe----------------------------------------------------------------------
######################
#     spe data 
######################

# Downloading spatially-resolved transcriptomics data
# EH9605 | spe 
spatial_hpc_spe <- myfiles[["EH9605"]]

# This is a SpatialExperiment object
spatial_hpc_spe

# Check sample info
head(colData(spatial_hpc_spe), 3)

# Check gene info
head(rowData(spatial_hpc_spe), 3)

# Access the original counts
assays(spatial_hpc_spe)$counts[1:3, 1:3]

# Access the log-normalized counts
assays(spatial_hpc_spe)$logcounts[1:3, 1:3]

# Access the reduced dimension methods
reducedDimNames(spatial_hpc_spe)

# Access the spatial coordinates
spatialCoordsNames(spatial_hpc_spe)

rm(spatial_hpc_spe)

## ----sce----------------------------------------------------------------------
######################
#     sce data 
######################

# Downloading single-nucleus RNA sequencing data
# EH9606 | sce
spatial_hpc_snrna_seq <- myfiles[["EH9606"]]

# This is a SingleCellExperiment object
spatial_hpc_snrna_seq

# Check sample info
head(colData(spatial_hpc_snrna_seq),3)

# Check gene info
head(rowData(spatial_hpc_snrna_seq), 3)

# Access the original counts
assays(spatial_hpc_snrna_seq)$counts[1:3, 1:3]

# Access the log-normalized counts
assays(spatial_hpc_snrna_seq)$logcounts[1:3, 1:3]

# Access the reduced dimension methods
reducedDimNames(spatial_hpc_snrna_seq)

## ----citation-----------------------------------------------------------------
citation("humanHippocampus2024")

## ----bib, echo=FALSE----------------------------------------------------------
library("RefManageR")

bib <- c(
    R = citation(),
    AnnotationHubData = citation("AnnotationHubData")[1],
    ExperimentHub = citation("ExperimentHub")[1],
    BiocStyle = citation("BiocStyle")[1],
    knitr = citation("knitr")[1],
    RefManageR = citation("RefManageR")[1],
    rmarkdown = citation("rmarkdown")[1],
    sessioninfo = citation("sessioninfo")[1],
    testthat = citation("testthat")[1]
)

## ----time, echo=FALSE---------------------------------------------------------
Sys.time()

## ----'sessionInfo', echo=FALSE----------------------------------------------------------------------------------------
# Session info
library(sessioninfo)
options(width = 120)
session_info()

## ----results = "asis", echo = FALSE, warning = FALSE, message = FALSE-------------------------------------------------
## Print bibliography
PrintBibliography(bib, .opts = list(hyperlink = "to.doc", style = "html"))

