// VerboseWorkSet.java, created Mon Jul 25 11:16:16 2005 by salcianu
// Copyright (C) 2005 Alexandru Salcianu <salcianu@alum.mit.edu>
// Licensed under the Modified BSD Licence; see COPYING for details.
package jpaul.DataStructs;

import java.io.Serializable;
import java.util.Collection;

/**
 * <code>VerboseWorkSet</code> - Debugging wrapper for a
 * <code>WorkSet</code>.  Prints helpful messages each time the state
 * of the <code>WorkSet</code> is modified (ex: adding/extracting
 * elements and when clearing the entire content of the
 * <code>WorkSet</code>.).
 * 
 * @author  Alexandru Salcianu - salcianu@alum.mit.edu
 * @version $Id: VerboseWorkSet.java,v 1.6 2006/03/14 02:29:31 salcianu Exp $ */
public class VerboseWorkSet<T> implements WorkSet<T>, Serializable {
    
    private static final long serialVersionUID = -974244175404086218L;

    /** Creates a <code>VerboseWorkSet</code>. 

	@param wSet Underlying <code>WorkSet</code>. 

	@param preffix String to be printed at the beginning of each
	output line.  This preffix allows the programmer to make the
	distinction between messages generated by different
	<code>VerboseWorkSet</code>s. */
    public VerboseWorkSet(WorkSet<T> wSet, String preffix) {
        this.wSet    = wSet;
	this.preffix = preffix;
    }

    private final WorkSet<T> wSet;
    private final String preffix;

    public boolean add(T elem) {
	if(wSet.add(elem)) {
	    System.out.println(preffix + "new Elem" + elem + "\t-> " + this);
	    return true;
	}
	return false;
    }

    public boolean addAll(Collection<T> elems) {
	boolean changed = false;
	for(T elem : elems) {
	    if(add(elem)) {
		changed = true;
	    }
	}
	return changed;
    }

    public T extract() {
	T elem = wSet.extract();
	System.out.println(preffix + "extracted " + elem + "\t-> " + this);
	return elem;
    }

    public void clear() {
	wSet.clear();
	System.out.println(preffix + "flush " + this);
    }

    public boolean isEmpty() {
	return wSet.isEmpty();
    }

    public boolean contains(T e) {
	return wSet.contains(e);
    }

    public int size() {
	return wSet.size();
    }

    public String toString() {
	return wSet.toString();
    }
   
}
