% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/functions-binning.R
\name{imputeLinInterpol}
\alias{imputeLinInterpol}
\title{Impute values for empty elements in a vector using linear interpolation}
\usage{
imputeLinInterpol(
  x,
  baseValue,
  method = "lin",
  distance = 1L,
  noInterpolAtEnds = FALSE
)
}
\arguments{
\item{x}{A numeric vector with eventual missing (\code{NA}) values.}

\item{baseValue}{The base value to which empty elements should be set. This
is only considered for \code{method = "linbase"} and corresponds to the
\code{profBinLinBase()}'s \code{baselevel} argument.}

\item{method}{One of \code{"none"}, \code{"lin"} or \code{"linbase"}.}

\item{distance}{For \code{method = "linbase"}: number of non-empty
neighboring element of an empty element that should be considered for
linear interpolation. See details section for more information.}

\item{noInterpolAtEnds}{For \code{method = "lin"}: Logical indicating
whether linear interpolation should also be performed at the ends of the
data vector (i.e. if missing values are present at the beginning or the
end of the vector).}
}
\value{
A numeric vector with empty values imputed based on the selected
\code{method}.
}
\description{
This function provides missing value imputation based on linear
interpolation and resembles some of the functionality of the
\code{profBinLin()} and \code{profBinLinBase()} functions deprecated from
version 1.51 on.
}
\details{
Values for NAs in input vector \code{x} can be imputed using methods
\code{"lin"} and \code{"linbase"}:

\if{html}{\out{<div class="sourceCode">}}\preformatted{`impute = "lin"` uses simple linear imputation to derive a value
for an empty element in input vector `x` from its neighboring
non-empty elements. This method is equivalent to the linear
interpolation in the `profBinLin` method. Whether interpolation is
performed if missing values are present at the beginning and end of
`x` can be set with argument `noInterpolAtEnds`. By default
interpolation is also performed at the ends interpolating from `0`
at the beginning and towards `0` at the end. For
`noInterpolAtEnds = TRUE` no interpolation is performed at both
ends replacing the missing values at the beginning and/or the end of
`x` with `0`.

`impute = "linbase"` uses linear interpolation to impute values for
empty elements within a user-definable proximity to non-empty elements
and setting the element's value to the `baseValue` otherwise. The
default for the `baseValue` is half of the smallest value in
`x` (`NA`s being removed). Whether linear interpolation based
imputation is performed for a missing value depends on the
`distance` argument. Interpolation is only performed if one of the
next `distance` closest neighbors to the current empty element has
a value other than `NA`. No interpolation takes place for
`distance = 0`, while `distance = 1` means that the value for
an empty element is interpolated from directly adjacent non-empty
elements while, if the next neighbors of the current empty element are
also `NA`, it's vale is set to `baseValue`.
This corresponds to the linear interpolation performed by the
`profBinLinBase` method. For more details see examples below.
}\if{html}{\out{</div>}}
}
\examples{
#######
## Impute missing values by linearly interpolating from neighboring
## non-empty elements
x <- c(3, NA, 1, 2, NA, NA, 4, NA, NA, NA, 3, NA, NA, NA, NA, 2)
imputeLinInterpol(x, method = "lin")
## visualize the interpolation:
plot(x = 1:length(x), y = x)
points(x = 1:length(x), y = imputeLinInterpol(x, method = "lin"), type = "l", col = "grey")

## If the first or last elements are NA, interpolation is performed from 0
## to the first non-empty element.
x <- c(NA, 2, 1, 4, NA)
imputeLinInterpol(x, method = "lin")
## visualize the interpolation:
plot(x = 1:length(x), y = x)
points(x = 1:length(x), y = imputeLinInterpol(x, method = "lin"), type = "l", col = "grey")

## If noInterpolAtEnds is TRUE no interpolation is performed at both ends
imputeLinInterpol(x, method = "lin", noInterpolAtEnds = TRUE)

######
## method = "linbase"
## "linbase" performs imputation by interpolation for empty elements based on
## 'distance' adjacent non-empty elements, setting all remaining empty elements
## to the baseValue
x <- c(3, NA, 1, 2, NA, NA, 4, NA, NA, NA, 3, NA, NA, NA, NA, 2)
## Setting distance = 0 skips imputation by linear interpolation
imputeLinInterpol(x, method = "linbase", distance = 0)

## With distance = 1 for all empty elements next to a non-empty element the value
## is imputed by linear interpolation.
xInt <- imputeLinInterpol(x, method = "linbase", distance = 1L)
xInt

plot(x = 1:length(x), y = x, ylim = c(0, max(x, na.rm = TRUE)))
points(x = 1:length(x), y = xInt, type = "l", col = "grey")

## Setting distance = 2L would cause that for all empty elements for which the
## distance to the next non-empty element is <= 2 the value is imputed by
## linear interpolation:
xInt <- imputeLinInterpol(x, method = "linbase", distance = 2L)
xInt

plot(x = 1:length(x), y = x, ylim = c(0, max(x, na.rm = TRUE)))
points(x = 1:length(x), y = xInt, type = "l", col = "grey")
}
\author{
Johannes Rainer
}
