% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/create_motif.R
\name{create_motif}
\alias{create_motif}
\alias{create_motif,missing-method}
\alias{create_motif,numeric-method}
\alias{create_motif,character-method}
\alias{create_motif,matrix-method}
\alias{create_motif,DNAStringSet-method}
\alias{create_motif,RNAStringSet-method}
\alias{create_motif,AAStringSet-method}
\alias{create_motif,BStringSet-method}
\title{Create a motif.}
\usage{
create_motif(input, alphabet, type = "PPM", name = "motif",
  pseudocount = 0, bkg, nsites, altname, family, organism, bkgsites, strand,
  pval, qval, eval, extrainfo, add.multifreq)

\S4method{create_motif}{missing}(input, alphabet, type = "PPM",
  name = "motif", pseudocount = 0, bkg, nsites, altname, family, organism,
  bkgsites, strand, pval, qval, eval, extrainfo, add.multifreq)

\S4method{create_motif}{numeric}(input, alphabet, type = "PPM",
  name = "motif", pseudocount = 0, bkg, nsites, altname, family, organism,
  bkgsites, strand, pval, qval, eval, extrainfo, add.multifreq)

\S4method{create_motif}{character}(input, alphabet, type = "PPM",
  name = "motif", pseudocount = 0, bkg, nsites, altname, family, organism,
  bkgsites, strand, pval, qval, eval, extrainfo, add.multifreq)

\S4method{create_motif}{matrix}(input, alphabet, type = "PPM",
  name = "motif", pseudocount = 0, bkg, nsites, altname, family, organism,
  bkgsites, strand, pval, qval, eval, extrainfo, add.multifreq)

\S4method{create_motif}{DNAStringSet}(input, alphabet, type = "PPM",
  name = "motif", pseudocount = 0, bkg, nsites, altname, family, organism,
  bkgsites, strand, pval, qval, eval, extrainfo, add.multifreq)

\S4method{create_motif}{RNAStringSet}(input, alphabet, type = "PPM",
  name = "motif", pseudocount = 0, bkg, nsites, altname, family, organism,
  bkgsites, strand, pval, qval, eval, extrainfo, add.multifreq)

\S4method{create_motif}{AAStringSet}(input, alphabet, type = "PPM",
  name = "motif", pseudocount = 0, bkg, nsites, altname, family, organism,
  bkgsites, strand, pval, qval, eval, extrainfo, add.multifreq)

\S4method{create_motif}{BStringSet}(input, alphabet, type = "PPM",
  name = "motif", pseudocount = 0, bkg, nsites, altname, family, organism,
  bkgsites, strand, pval, qval, eval, extrainfo, add.multifreq)
}
\arguments{
\item{input}{\code{character}, \code{numeric}, \code{matrix},
\code{\link{XStringSet}}, or \code{missing}.}

\item{alphabet}{\code{character(1)} One of \code{c('DNA', 'RNA', 'AA')},
or a combined string representing the letters. If no alphabet is
provided then it will try and guess the alphabet from the input.}

\item{type}{\code{character(1)} One of \code{c('PCM', 'PPM', 'PWM', 'ICM')}.}

\item{name}{\code{character(1)} Motif name.}

\item{pseudocount}{\code{numeric(1)} Correction to be applied to prevent \code{-Inf}
from appearing in PWM matrices. Defaults to 0.}

\item{bkg}{\code{numeric} A vector of probabilities, each between 0 and 1. If
higher order backgrounds are provided, then the elements of the vector
must be named. If unnamed, then the order of probabilities must be in the
same order as the alphabetically sorted sequence alphabet.}

\item{nsites}{\code{numeric(1)} Number of sites the motif was constructed from. If
blank, then \code{create_motif()} will guess the appropriate number if possible.
To prevent this, provide \code{nsites = numeric()}.}

\item{altname}{\code{character(1)} Alternate motif name.}

\item{family}{\code{character(1)} Transcription factor family.}

\item{organism}{\code{character(1)} Species of origin.}

\item{bkgsites}{\code{numeric(1)} Total number of sites used to find the motif.}

\item{strand}{\code{character(1)} Whether the motif is specific to a certain strand.
Acceptable strands are '+', '-', and '+-' (to represent both strands). Note
that '-+' and '*' can also be provided to represent both strands, but the
final strand in the \code{universalmotif} object will be set to '+-'.}

\item{pval}{\code{numeric(1)} P-value associated with motif.}

\item{qval}{\code{numeric(1)} Adjusted P-value associated with motif.}

\item{eval}{\code{numeric(1)} E-value associated with motif.}

\item{extrainfo}{\code{character} Any other extra information, represented as
a named character vector.}

\item{add.multifreq}{\code{numeric} If the motif is created from a set of
sequences, then the \code{\link[=add_multifreq]{add_multifreq()}} function can be
run at the same time (with \code{RC = FALSE}).}
}
\value{
\linkS4class{universalmotif} object.
}
\description{
Create a motif from a set of sequences, a matrix, or generate a random
motif. See the "Motif import, export and manipulation" vignette for details.
}
\details{
The aim of this function is provide an easy interface to creating
\linkS4class{universalmotif} motifs, as an alternative to the
default class constructor (i.e. \code{new('universalmotif', name=...)}).
See examples for potential use cases.

Note: when generating random motifs, the \code{nsites} slot is also given a
random value.

See the \code{examples} section for more info on motif creation.
}
\section{Methods (by class)}{
\itemize{
\item \code{create_motif(missing)}: Create a random motif of length 10.

\item \code{create_motif(numeric)}: Create a random motif with a specified length.

\item \code{create_motif(character)}: Create motif from a consensus string.

\item \code{create_motif(matrix)}: Create motif from a matrix.

\item \code{create_motif(DNAStringSet)}: Create motif from a \code{\link{DNAStringSet}}.

\item \code{create_motif(RNAStringSet)}: Create motif from a \code{\link{RNAStringSet}}.

\item \code{create_motif(AAStringSet)}: Create motif from a \code{\link{AAStringSet}}.

\item \code{create_motif(BStringSet)}: Create motif from a \code{\link{BStringSet}}.

}}
\examples{
##### create motifs from a single string

# Motif is by default generated as a PPM: change final type as desired
DNA.motif <- create_motif("TATAWAW")
DNA.motif <- create_motif("TATAWAW", type = "PCM")

# Nsites will be set to the number of input sequences unless specified or
# a single string is used as input
DNA.motif <- create_motif("TTTTTTT", nsites = 10)

# Ambiguity letters can be used:
DNA.motif <- create_motif("TATAWAW")
DNA.motif <- create_motif("NNVVWWAAWWDDN")

# Be careful about setting nsites when using ambiguity letters!
DNA.motif <- create_motif("NNVVWWAAWWDDN", nsites = 1)

RNA.motif <- create_motif("UUUCCG")

# 'create_motif' will try to detect the alphabet type; this can be
# unreliable for AA and custom alphabets as DNA and RNA alphabets are
# detected first
AA.motif <- create_motif("AVLK", alphabet = "AA")

custom.motif <- create_motif("QWER", alphabet = "QWER")
# Specify custom alphabet
custom.motif <- create_motif("QWER", alphabet = "QWERASDF")

###### Create motifs from multiple strings of equal length

DNA.motif <- create_motif(c("TTTT", "AAAA", "AACC", "TTGG"), type = "PPM")
DNA.motif <- create_motif(c("TTTT", "AAAA", "AACC", "TTGG"), nsites = 20)
RNA.motif <- create_motif(c("UUUU", "AAAA", "AACC", "UUGG"), type = "PWM")
AA.motif <- create_motif(c("ARNDCQ", "EGHILK", "ARNDCQ"), alphabet = "AA")
custom.motif <- create_motif(c("POIU", "LKJH", "POIU", "CVBN"),
                             alphabet = "POIULKJHCVBN")

# Ambiguity letters are only allowed for single consensus strings: the
# following fails
\dontrun{
create_motif(c("WWTT", "CCGG"))
create_motif(c("XXXX", "XXXX"), alphabet = "AA")
}

##### Create motifs from XStringSet objects

library(Biostrings)

DNA.set <- DNAStringSet(c("TTTT", "AAAA", "AACC", "TTGG"))
DNA.motif <- create_motif(DNA.set)
RNA.set <- RNAStringSet(c("UUUU", "AACC", "UUCC"))
RNA.motif <- create_motif(RNA.set)
AA.set <- AAStringSet(c("VVVLLL", "AAAIII"))
AA.motif <- create_motif(AA.set)

# Custom motifs can be created from BStringSet objects
B.set <- BStringSet(c("QWER", "ASDF", "ZXCV", "TYUI"))
custom.motif <- create_motif(B.set)

##### Create motifs with filled 'multifreq' slot

DNA.motif.k2 <- create_motif(DNA.set, add.multifreq = 2)

##### Create motifs from matrices

mat <- matrix(c(1, 1, 1, 1,
                2, 0, 2, 0,
                0, 2, 0, 2,
                0, 0, 0, 0),
                nrow = 4, byrow = TRUE)
DNA.motif <- create_motif(mat, alphabet = "DNA")
RNA.motif <- create_motif(mat, alphabet = "RNA", nsites = 20)
custom.motif <- create_motif(mat, alphabet = "QWER")

# Specify custom alphabet
custom.motif <- create_motif(mat, alphabet = "QWER")

# Alphabet can be detected from rownames
rownames(mat) <- DNA_BASES
DNA.motif <- create_motif(mat)
rownames(mat) <- c("Q", "W", "E", "R")
custom.motif <- create_motif(mat)

# Matrices can also be used as input
mat.ppm <- matrix(c(0.1, 0.1, 0.1, 0.1,
                    0.5, 0.5, 0.5, 0.5,
                    0.1, 0.1, 0.1, 0.1,
                    0.3, 0.3, 0.3, 0.3),
                    nrow = 4, byrow = TRUE)

DNA.motif <- create_motif(mat.ppm, alphabet = "DNA", type = "PPM")

##### Create random motifs

# These are generated as PPMs with 10 positions

DNA.motif <- create_motif()
RNA.motif <- create_motif(alphabet = "RNA")
AA.motif <- create_motif(alphabet = "AA")
custom.motif <- create_motif(alphabet = "QWER")

# The number of positions can be specified

DNA.motif <- create_motif(5)

# If the background frequencies are not provided, they are generated
# using `rpois`; positions are created using `rdirichlet(1, bkg)`.
# (calling `create_motif()` creates motifs with an average
# positional IC of 1)

DNA.motif <- create_motif(bkg = c(0.3, 0.2, 0.2, 0.3))
DNA.motif <- create_motif(10, bkg = c(0.1, 0.4, 0.4, 0.1))

}
\seealso{
\code{\link[=convert_type]{convert_type()}}, \code{\link[=add_multifreq]{add_multifreq()}}, \code{\link[=create_sequences]{create_sequences()}},
\code{\link[=shuffle_motifs]{shuffle_motifs()}}.
}
\author{
Benjamin Jean-Marie Tremblay, \email{benjamin.tremblay@uwaterloo.ca}
}
