#' @title Normalization by loess method
#'
#' @description Apply loess normalization to data
#'
#' @param dataMatrix Data frame or numeric matrix. Columns are plates, and rows are plate wells.
#' @param dataRows,dataCols Optional integer vector. Indicate which row/column numbers from the dataMatrix should be normalized. If NULL then all rows/columns from the dataMatrix are used.
#' @param plateRows,plateCols Number of rows/columns in plate.
#'
#' @family normalization methods
#'
#' @details Loess normalization adjusts each well by the fitted row and column values generated by calculating the loess curve for each row and column.
#'
#' @include internal.R normSPAWN.R
#'
#' @references \href{http://www.ncbi.nlm.nih.gov/pmc/articles/PMC3117325/}{Baryshnikova et al. (2010). Quantitative analysis of fitness and genetic interactions in yeast on a genome scale. Nature Methods, 7(12), 1017-1024.}
#'
#' @return Numeric matrix of normalized data in the same format as dataMatrix
#'
#' @note For information on how to arrange your dataset for dataMatrix, please see (\code{\link{ex_dataMatrix}})
#'
#' @examples
#' ## load dataset
#' data(ex_dataMatrix)
#'
#' ## apply Loess method
#' ex_normMatrix <- normLoess(dataMatrix = ex_dataMatrix, dataCols = 5:10,
#' plateRows = 8, plateCols = 10)
#'
#' @export
normLoess <- function(dataMatrix, plateRows, plateCols, dataRows = NULL, dataCols = NULL) {
    
    dataMatrix <- sightsCheck(dataMatrix, "data", dataRows, dataCols, plateRows, plateCols)
    
    outMat <- apply(dataMatrix, 2, loessMe, plateRows, plateCols)
    outMat <- matrix(apply(outMat, 2, myRobZ), nrow = nrow(dataMatrix), ncol = ncol(dataMatrix), dimnames = dimnames(dataMatrix))
    
    message("Completed Loess normalization")
    message("Number of plates = ", ncol(dataMatrix))
    message("Number of plate wells = ", nrow(dataMatrix))
    return(outMat)
}
