% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RDP.R
\name{rdp}
\alias{rdp}
\alias{RDP}
\alias{predict}
\alias{predict.RDPClassifier}
\alias{print.RDPClassifier}
\alias{trainRDP}
\alias{removeRDP}
\title{Ribosomal Database Project (RDP) Classifier for 16S rRNA}
\usage{
rdp(dir = NULL)

\method{predict}{RDPClassifier}(object, newdata, confidence = 0.8, rdp_args = "", verbose = FALSE, ...)

trainRDP(x, dir = "classifier", rank = "genus", verbose = FALSE)

removeRDP(object)
}
\arguments{
\item{dir}{directory where the classifier information is stored.}

\item{object}{a RDPClassifier object.}

\item{newdata}{new data to be classified as a \link[Biostrings:XStringSet-class]{Biostrings::DNAStringSet}.}

\item{confidence}{numeric; minimum confidence level for classification.
Results with lower confidence are replaced by NAs. Set to 0 to disable.}

\item{rdp_args}{additional RDP arguments for classification (e.g.,
\code{"-minWords 5"} to set the minimum number of words for each bootstrap trial.).
See RDP documentation.}

\item{verbose}{logical; print additional information.}

\item{...}{additional arguments (currently unused).}

\item{x}{an object of class \link[Biostrings:XStringSet-class]{Biostrings::DNAStringSet} with the 16S rRNA sequences for
training.}

\item{rank}{Taxonomic rank at which the classification is learned.}
}
\value{
\code{rdp()} and \code{trainRDP()} return a \code{RDPClassifier} object.

\code{predict()} returns a data.frame containing the classification results
for each sequence (rows). The data.frame has an attribute called
\code{"confidence"} with a matrix containing the confidence values.
}
\description{
Use the RDP classifier (Wang et al, 2007) to classify 16S rRNA sequences.
This package contains
currently RDP version 2.14 released in August 2023. The associated data
package \code{rRDPData} contains models trained on
the bacterial and archaeal taxonomy training set No. 19 (see Wang and
Cole, 2024).
}
\details{
RDP is a naive Bayes classifier using 8-mers as features.

\code{rdp()} creates a default classifier trained with the data shipped with
RDP.  Alternatively, a directory with the data for an existing classifier
(created with \code{trainRDP()}) can be supplied.

\code{trainRDP()} creates a new classifier for the data in \code{x} and
stores the classifier information in \code{dir}. The data in \code{x} needs
to have annotations in the following format:

\code{"<ID> <Kingdom>;<Phylum>;<Class>;<Order>;<Family>;<Genus>"}

A created classifier can be removed with \code{removeRDP()}. This will
remove the directory which stores the classifier information.

The data for the default 16S rRNA classifier can be found in package
\code{rRDPData}.
}
\examples{
### Use the default classifier
seq <- readRNAStringSet(system.file("examples/RNA_example.fasta",
    package = "rRDP"
))

## shorten names
names(seq) <- sapply(strsplit(names(seq), " "), "[", 1)
seq

## use rdp for classification (this needs package rRDPData installed)
## > BiocManager::install("rRDPData")

cl_16S <- rdp()
cl_16S

pred <- predict(cl_16S, seq)
pred

attr(pred, "confidence")

### Train a custom RDP classifier on new data
trainingSequences <- readDNAStringSet(
    system.file("examples/trainingSequences.fasta", package = "rRDP")
)

customRDP <- trainRDP(trainingSequences)
customRDP

testSequences <- readDNAStringSet(
    system.file("examples/testSequences.fasta", package = "rRDP")
)
predict(customRDP, testSequences)

## clean up
removeRDP(customRDP)
}
\references{
Hahsler M, Nagar A (2020). "rRDP: Interface to the RDP
Classifier." R Package, Bioconductor.
\doi{10.18129/B9.bioc.rRDP}.

RDP classifier software:
\url{https://sourceforge.net/projects/rdp-classifier/}

Qiong Wang, George M. Garrity, James M. Tiedje and James R. Cole. Naive
Bayesian Classifier for Rapid Assignment of rRNA Sequences into the New
Bacterial Taxonomy, Appl. Environ. Microbiol. August 2007 vol. 73 no. 16
5261-5267. \doi{10.1128/AEM.00062-07}

Qiong W. and Cole J.R. Updated RDP taxonomy and RDP Classifier for more
accurate taxonomic classification, Microbial Ecology,
Announcement, 4 March 2024. \doi{https://doi.org/10.1128/mra.01063-23}
}
\keyword{model}
