% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calculateRelativeFC.R
\name{calculateRelativeFC}
\alias{calculateRelativeFC}
\title{Calculate logFCs relative to WT using edgeR}
\usage{
calculateRelativeFC(
  se,
  design,
  coef = NULL,
  contrast = NULL,
  WTrows = NULL,
  selAssay = "counts",
  pseudocount = 1,
  method = "edgeR",
  normMethod = ifelse(is.null(WTrows), "TMM", "sum")
)
}
\arguments{
\item{se}{SummarizedExperiment object.}

\item{design}{Design matrix. The rows of the design matrix must be in the
same order as the columns in \code{se}.}

\item{coef}{Coefficient(s) to test with edgeR or limma.}

\item{contrast}{Numeric contrast to test with edgeR or limma.}

\item{WTrows}{Vector of row names that will be used as the reference when
calculating logFCs and statistics. If more than one value is provided, 
the sum of the corresponding counts is used to generate offsets. If 
\code{NULL}, offsets will be defined as the effective library sizes 
(using TMM normalization factors).}

\item{selAssay}{Assay to select from \code{se} for the analysis.}

\item{pseudocount}{Pseudocount to add when calculating log-fold changes.}

\item{method}{Either 'edgeR' or 'limma'. If set to 'limma', voom is used to
transform the counts and estimate observation weights before applying
limma. In this case, the results also contain the standard errors of the
logFCs.}

\item{normMethod}{Character scalar indicating which normalization method
should be used to calculate size factors. Should be either \code{"TMM"}
or \code{"csaw"} when \code{WTrows} is \code{NULL}, and \code{"geomean"}
or \code{"sum"} when \code{WTrows} is provided.}
}
\value{
A \code{data.frame} with output from the statistical testing 
framework (edgeR or limma).
}
\description{
Calculate logFCs and associated p-values for a given comparison, using
either limma or the Negative Binomial quasi-likelihood framework of edgeR.
The observed counts for the WT variants can be used as offsets in the model.
}
\examples{
library(SummarizedExperiment)
se <- readRDS(system.file("extdata", "GSE102901_cis_se.rds",
                          package = "mutscan"))[1:200, ]
design <- model.matrix(~ Replicate + Condition,
                       data = colData(se))
                              
## Calculate "absolute" log-fold changes with edgeR
res <- calculateRelativeFC(se, design, coef = "Conditioncis_output", 
                           method = "edgeR")
head(res)
## Calculate log-fold changes relative to the WT sequence with edgeR
stopifnot("f.0.WT" \%in\% rownames(se))
res <- calculateRelativeFC(se, design, coef = "Conditioncis_output", 
                           method = "edgeR", WTrows = "f.0.WT")
head(res)

## Calculate "absolute" log-fold changes with limma
res <- calculateRelativeFC(se, design, coef = "Conditioncis_output", 
                           method = "limma")
head(res)
## Calculate log-fold changes relative to the WT sequence with limma
stopifnot("f.0.WT" \%in\% rownames(se))
res <- calculateRelativeFC(se, design, coef = "Conditioncis_output", 
                           method = "limma", WTrows = "f.0.WT")
head(res)

}
\author{
Charlotte Soneson, Michael Stadler
}
