#' Reorder the data frame for differential analysis, ensuring control verse 
#' treatment comparison
#' 
#' 
#' @description Re-organise the working data frame, placing control samples 
#' before treatment samples. This ensures differential analysis comparison 
#' between controls and treatment are in the correct arrangement. 
#'
#' @param data data.frame; generated by [mobileRNA::RNAimport()]  
#'
#'
#'@param controls character; vector of control condition sample names. 
#'
#' 
#' @return A re-ordered/re-organised working data frame with control samples 
#' after the 5 cluster information columns, and treatment sample columns after
#' the control sample columns. 
#' @examples
#' # load data 
#' data("sRNA_data")
#' 
#' controlReps <- c("selfgraft_1", "selfgraft_2", "selfgraft_3")
#' 
#' reorder_df <- RNAreorder(data = sRNA_data, controls = controlReps)
#' 
#' @export
RNAreorder <- function(data, controls) {
  if (!is.data.frame(data)) {
    stop("Input 'data' must be a data frame.")
  }
  if ( is.null(controls) || !base::inherits(controls, "character")) {
    stop("group must be an vector of characters to specify the sample 
         replicates within the control condition. ")
  }
  # Identify the column indices that contain keywords from the 'controls' vector
  control_indices <- vapply(controls, function(keyword) grep(keyword, 
                                                             colnames(data)),
                            FUN.VALUE = integer(length(controls)+1))
  # Remove NA indices
  control_indices <- control_indices[!is.na(control_indices)]
  # to place columns after the first 5
  control_indices <- control_indices[control_indices > 5]
  # Reorder the columns
  range <- seq_len(ncol(data) - 5) + 5
  new_order <- c(seq_len(5), control_indices, base::setdiff(range, control_indices))
  data <- data[, new_order]
  return(data)
}
