% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/test-only-hierarchy.R
\name{test_only_hierarchy}
\alias{test_only_hierarchy}
\title{Hierarchical Testing}
\usage{
test_only_hierarchy(x, y, dendr, res.multisplit, clvar = NULL,
  family = c("gaussian", "binomial"), alpha = 0.05,
  global.test = TRUE, agg.method = c("Tippett", "Stouffer"),
  verbose = FALSE, sort.parallel = TRUE, parallel = c("no",
  "multicore", "snow"), ncpus = 1L, cl = NULL, check.input = TRUE,
  unique.colnames.x = NULL)
}
\arguments{
\item{x}{a matrix or list of matrices for multiple data sets. The matrix or
matrices have to be of type numeric and are required to have column names
/ variable names. The rows and the columns represent the observations and
the variables, respectively.}

\item{y}{a vector, a matrix with one column, or list of the aforementioned
objects for multiple data sets. The vector, vectors, matrix, or matrices
have to be of type numeric. For \code{family = "binomial"}, the response
is required to be a binary vector taking values 0 and 1.}

\item{dendr}{the output of one of the functions
\code{\link{cluster_var}} or \code{\link{cluster_position}}.}

\item{res.multisplit}{the output of the function
\code{\link{multisplit}}.}

\item{clvar}{a matrix or list of matrices of control variables.}

\item{family}{a character string naming a family of the error distribution;
either \code{"gaussian"} or \code{"binomial"}.}

\item{alpha}{the significant level at which the FWER is controlled.}

\item{global.test}{a logical value indicating whether the global test should
be performed.}

\item{agg.method}{a character string naming an aggregation method which
aggregates the p-values over the different data sets for a given cluster;
either \code{"Tippett"} (Tippett's rule) or \code{"Stouffer"}
(Stouffer's rule). This argument is only relevant if multiple data sets
are specified in the function call.}

\item{verbose}{a logical value indicating whether the progress of the computation
should be printed in the console.}

\item{sort.parallel}{a logical indicating whether the values are sorted with respect to
the size of the block. This can reduce the run time for parallel computation.}

\item{parallel}{type of parallel computation to be used. See the 'Details' section.}

\item{ncpus}{number of processes to be run in parallel.}

\item{cl}{an optional \strong{parallel} or \strong{snow} cluster used if
\code{parallel = "snow"}. If not supplied, a cluster on the local machine is created.}

\item{check.input}{a logical value indicating whether the function should
check the input. This argument is used to call
\code{\link{test_only_hierarchy}} within
\code{\link{test_hierarchy}}.}

\item{unique.colnames.x}{a character vector containing the unique column
names of \code{x}. This argument is used to call
\code{\link{test_only_hierarchy}} within
\code{\link{test_hierarchy}}.}
}
\value{
The returned value is an object of class \code{"hierT"}, consisting of
two elements, the result of the multi-sample splitting step
\code{"res.multisplit"} and the result of the hierarchical testing
\code{"res.hierarchy"}.

The result of the multi-sample splitting step is a list with number of
elements corresponding to the number of data sets. Each element
(corresponding to a data set) contains a list with two matrices. The first
matrix contains the indices of the second half of variables (which were
not used to select the variables). The second matrix contains the column
names / variable names of the selected variables.

The result of the hierarchical testing is a data frame of significant
clusters with the following columns:
\item{block}{\code{NA} or the name of the block if the significant cluster
is a subcluster of the block or is the block itself.}
\item{p.value}{The p-value of the significant cluster.}
\item{significant.cluster}{The column names of the members of the significant
cluster.}

There is a \code{print} method for this class; see
\code{\link{print.hierT}}.
}
\description{
Hierarchical testing given the output of the function
\code{\link{multisplit}}.
}
\details{
The function \code{\link{test_only_hierarchy}} requires the output
of one of the functions \code{\link{cluster_var}} or
\code{\link{cluster_position}} as an input (argument \code{dendr}).
Furthermore it requires the output of the function
\code{\link{multisplit}} as an input (argument \code{res.multisplit}).
Hierarchical testing is performed by going top down through the hierarchical
tree. Testing only continues if at least one child of a given cluster is significant.

If the argument \code{block} was supplied for the building
of the hierarchical tree (i.e. in the function call of either
\code{\link{cluster_var}} or
\code{\link{cluster_position}}), i.e. the second level of the
hierarchical tree was given, the hierarchical testing step can be run in
parallel across the different blocks by specifying the arguments
\code{parallel} and \code{ncpus}. There is an optional argument \code{cl} if
\code{parallel = "snow"}. There are three possibilities to set the
argument \code{parallel}: \code{parallel = "no"} for serial evaluation
(default), \code{parallel = "multicore"} for parallel evaluation
using forking, and \code{parallel = "snow"} for parallel evaluation
using a parallel socket cluster. It is recommended to select
\code{\link{RNGkind}("L'Ecuyer-CMRG")} and set a seed to ensure that
the parallel computing of the package \code{hierinf} is reproducible.
This way each processor gets a different substream of the pseudo random
number generator stream which makes the results reproducible if the arguments
(as \code{sort.parallel} and \code{ncpus}) remain unchanged. See the vignette
or the reference for more details.

Note that if Tippett's aggregation method is applied for multiple data
sets, then very small p-values are set to machine precision. This is
due to rounding in floating point arithmetic.
}
\examples{
n <- 200
p <- 500
library(MASS)
set.seed(3)
x <- mvrnorm(n, mu = rep(0, p), Sigma = diag(p))
colnames(x) <- paste0("Var", 1:p)
beta <- rep(0, p)
beta[c(5, 20, 46)] <- 1
y <- x \%*\% beta + rnorm(n)

dendr1 <- cluster_var(x = x)
set.seed(76)
res.multisplit1 <- multisplit(x = x, y = y, family = "gaussian")
sign.clusters1 <- test_only_hierarchy(x = x, y = y, dendr = dendr1,
                                      res.multisplit = res.multisplit1,
                                      family = "gaussian")

## With block
# The column names of the data frame block are optional.
block <- data.frame("var.name" = paste0("Var", 1:p),
                    "block" = rep(c(1, 2), each = p/2),
                    stringsAsFactors = FALSE)
dendr2 <- cluster_var(x = x, block = block)
# The output res.multisplit1 can be used since the multi-sample
# step is the same with or without blocks.
sign.clusters2 <- test_only_hierarchy(x = x, y = y, dendr = dendr2,
                                      res.multisplit = res.multisplit1,
                                      family = "gaussian")

# Access part of the object
sign.clusters2$res.hierarchy[, "block"]
sign.clusters2$res.hierarchy[, "p.value"]
# Column names or variable names of the significant cluster in the first row.
sign.clusters2$res.hierarchy[[1, "significant.cluster"]]

}
\references{
Renaux, C. et al. (2018), Hierarchical inference for genome-wide
association studies: a view on methodology with software. (arXiv:1805.02988)
}
\seealso{
\code{\link{cluster_var}},
\code{\link{cluster_position}},
\code{\link{multisplit}},
\code{\link{test_hierarchy}}, and
\code{\link{compute_r2}}.
}
