test_that("getRegionsFromTxDb() runs", {
    requireNamespace("txdbmaker")
    requireNamespace("GenomicFeatures")
    txdb <- AnnotationDbi::loadDb(system.file("extdata", "hg19sub.sqlite", package = "eisaR"))
    regL <- getRegionsFromTxDb(txdb)

    # arguments
    expect_error(getRegionsFromTxDb(seq.int(10L)))
    expect_error(getRegionsFromTxDb(txdb, exonExt = "a"))
    expect_error(getRegionsFromTxDb(txdb, strandedData = "a"))

    # expected results
    expect_type(regL, "list")
    expect_length(regL, 2L)
    expect_equivalent(lengths(regL), c(20L, 4L))
    expect_true(all(countOverlaps(regL$exons) == 1L))

    # same for EnsDb object
    # ... load GRanges with annotation (Y)
    load(system.file("YGRanges.RData", package = "ensembldb"))
    # ... create EnsDb
    suppressWarnings(eDB <- ensembldb::ensDbFromGRanges(Y, path = tempdir(), version = 75L,
                                                        organism = "Homo_sapiens"))
    edb <- ensembldb::EnsDb(eDB)
    # ... create corresponding TxDb
    suppressWarnings(txdb <- txdbmaker::makeTxDbFromGRanges(Y))
    # ... compare annotation
    expect_length(GenomicFeatures::genes(edb),
                  length(GenomicFeatures::genes(txdb)))
    expect_length(GenomicFeatures::exons(edb),
                  length(GenomicFeatures::exons(txdb)))
    # ... compare results
    regL1 <- getRegionsFromTxDb(edb)
    regL2 <- getRegionsFromTxDb(txdb)
    expect_identical(regL1, regL2)

    # clean up
    AnnotationDbi::dbFileDisconnect(AnnotationDbi::dbconn(txdb))
})

test_that("getRegionsFromTbx() fails when required packages are missing", {
    # these tests assume that all non-base packages are installed in
    # .Library.site and will fail if this is not the case (e.g. on BioC builders)
    skip_on_bioc()
    # also skip on r-universe (https://docs.r-universe.dev/publish/troubleshoot-build.html#how-to-know-whether-tests-are-run-on-r-universe)
    skip_if_not(identical(Sys.getenv("MY_UNIVERSE"), ""))

    # set new lib paths
    # ... unload ns
    unloadNamespace("ensembldb")
    unloadNamespace("txdbmaker")
    unloadNamespace("GenomicFeatures")
    withr::with_temp_libpaths({
        dummy <- list()
        class(dummy) <- "TxDb"
        expect_error(getRegionsFromTxDb(dummy))
    }, action = "replace")
    # ... reload ns
    requireNamespace("GenomicFeatures")
    requireNamespace("txdbmaker")
    requireNamespace("ensembldb")
})
