% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/assess_immune_response.R
\name{assess_immune_response}
\alias{assess_immune_response}
\title{Assess easier score as predictor of patients'
immune response}
\usage{
assess_immune_response(
  predictions_immune_response = NULL,
  patient_response = NULL,
  RNA_tpm = NULL,
  select_gold_standard = NULL,
  TMB_values = NULL,
  easier_with_TMB = "none",
  weight_penalty = NULL,
  verbose = TRUE
)
}
\arguments{
\item{predictions_immune_response}{list containing the predictions
for each quantitative descriptor and for each task. This is the
output from \code{predict_immune_response}.}

\item{patient_response}{character vector with two factors
(Non-responders = NR, Responders = R).}

\item{RNA_tpm}{numeric matrix of patients' gene expression data as
tpm values.}

\item{select_gold_standard}{character string with names of scores
of immune response to be computed. Default scores are computed for:
"CYT", "Roh_IS", "chemokines", "Davoli_IS", "IFNy", "Ayers_expIS",
"Tcell_inflamed", "RIR", "TLS".}

\item{TMB_values}{numeric vector containing patients' tumor mutational
burden (TMB) values.}

\item{easier_with_TMB}{character string indicating which approach
should be used to integrate easier with TMB. If \code{TMB_values}
provided, one of "weighted_average" (default) or "penalized_score".
If \code{TMB_values} not provided, default is "none".}

\item{weight_penalty}{integer value from 0 to 1, which is used to
define the weight or penalty for combining easier and TMB scores based
on a weighted average or penalized score, in order to derive a score of
patient's likelihood of immune response. The default value is 0.5.}

\item{verbose}{logical flag indicating whether to display messages
about the process.}
}
\value{
When \code{patient_response} is provided, a roc curve plot and
a bar plot that displays the average (across tasks) area under the ROC
curve (AUC) values is returned. If \code{patient_response} is not provided,
the easier score is represented as box plots (10 tasks) for each patient.

When \code{patient_response} is provided and \code{easier_with_TMB = weighted_average}
or \code{easier_with_TMB = penalized_score}, an scatter plot that shows the AUC
values of the integrated approach, easier score and TMB is returned.
If in this case, \code{patient_response} is not provided, the integrated score
is represented as a dot plot for each patient.
}
\description{
Evaluates the predictive performance of easier score as
predictor of patients' immune response. This is done for each
quantitative descriptor, an ensemble descriptor based on
the average of the individual ones, and the gold standard scores.
If provided, tumor mutational burden (TMB) is also used as
predictor for comparison. Since both immune response and TMB are
essential for effective immunotherapy response, an integrated
score is calculated given two different approaches based on a
applying either a weighted average or penalty to patients' easier
score depending on their TMB category.
}
\examples{
# using a SummarizedExperiment object
library(SummarizedExperiment)
# Using example exemplary dataset (Mariathasan et al., Nature, 2018)
# from easierData. Original processed data is available from
# IMvigor210CoreBiologies package.
library("easierData")

dataset_mariathasan <- easierData::get_Mariathasan2018_PDL1_treatment()
RNA_tpm <- assays(dataset_mariathasan)[["tpm"]]
cancer_type <- metadata(dataset_mariathasan)[["cancertype"]]

# Select a subset of patients to reduce vignette building time.
pat_subset <- c(
  "SAM76a431ba6ce1", "SAMd3bd67996035", "SAMd3601288319e",
  "SAMba1a34b5a060", "SAM18a4dabbc557"
)
RNA_tpm <- RNA_tpm[, colnames(RNA_tpm) \%in\% pat_subset]

# Computation of TF activity (Garcia-Alonso et al., Genome Res, 2019)
tf_activities <- compute_TF_activity(
  RNA_tpm = RNA_tpm
)

# Predict patients' immune response
predictions <- predict_immune_response(
  tfs = tf_activities,
  cancer_type = cancer_type,
  verbose = TRUE
)

# retrieve clinical response
patient_ICBresponse <- colData(dataset_mariathasan)[["BOR"]]
names(patient_ICBresponse) <- colData(dataset_mariathasan)[["pat_id"]]

# retrieve TMB
TMB <- colData(dataset_mariathasan)[["TMB"]]
names(TMB) <- colData(dataset_mariathasan)[["pat_id"]]

patient_ICBresponse <- patient_ICBresponse[names(patient_ICBresponse) \%in\% pat_subset]
TMB <- TMB[names(TMB) \%in\% pat_subset]

# Assess patient-specific likelihood of response to ICB therapy
output_eval_with_resp <- assess_immune_response(
  predictions_immune_response = predictions,
  patient_response = patient_ICBresponse,
  RNA_tpm = RNA_tpm,
  select_gold_standard = "IFNy",
  TMB_values = TMB,
  easier_with_TMB = "weighted_average",
)
\donttest{

RNA_counts <- assays(dataset_mariathasan)[["counts"]]
RNA_counts <- RNA_counts[, colnames(RNA_counts) \%in\% pat_subset]

# Computation of cell fractions  (Finotello et al., Genome Med, 2019)
cell_fractions <- compute_cell_fractions(RNA_tpm = RNA_tpm)

# Computation of pathway scores (Holland et al., BBAGRM, 2019;
# Schubert et al., Nat Commun, 2018)
pathway_activities <- compute_pathway_activity(
  RNA_counts = RNA_counts,
  remove_sig_genes_immune_response = TRUE
)

# Computation of ligand-receptor pair weights
lrpair_weights <- compute_LR_pairs(
  RNA_tpm = RNA_tpm,
  cancer_type = "pancan"
)

# Computation of cell-cell interaction scores
ccpair_scores <- compute_CC_pairs(
  lrpairs = lrpair_weights,
  cancer_type = "pancan"
)

# Predict patients' immune response
predictions <- predict_immune_response(
  pathways = pathway_activities,
  immunecells = cell_fractions,
  tfs = tf_activities,
  lrpairs = lrpair_weights,
  ccpairs = ccpair_scores,
  cancer_type = cancer_type,
  verbose = TRUE
)

# Assess patient-specific likelihood of response to ICB therapy
output_eval_with_resp <- assess_immune_response(
  predictions_immune_response = predictions,
  patient_response = patient_ICBresponse,
  RNA_tpm = RNA_tpm,
  TMB_values = TMB,
  easier_with_TMB = "weighted_average",
)
}
}
