% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/as_Seurat.R
\name{as_Seurat}
\alias{as_Seurat}
\title{Convert an \code{AnnData} to a \code{Seurat}}
\usage{
as_Seurat(
  adata,
  assay_name = "RNA",
  x_mapping = NULL,
  layers_mapping = TRUE,
  object_metadata_mapping = TRUE,
  assay_metadata_mapping = TRUE,
  reduction_mapping = TRUE,
  graph_mapping = TRUE,
  misc_mapping = TRUE
)
}
\arguments{
\item{adata}{The \code{AnnData} object to convert.}

\item{assay_name}{Name of the assay to be created in the new \code{Seurat} object}

\item{x_mapping}{A string specifying the name of the layer in the resulting
\code{Seurat} object where the data in the \code{X} slot of \code{adata} will be mapped to}

\item{layers_mapping}{A named vector where names are names of \code{Layers} in the
resulting \code{Seurat} object and values are keys of \code{layers} in \code{adata}. See
below for details.}

\item{object_metadata_mapping}{A named vector where names are cell metadata
columns in the resulting \code{Seurat} object and values are columns of \code{obs} in
\code{adata}. See below for details.}

\item{assay_metadata_mapping}{A named vector where names are variable
metadata columns in the assay of the resulting \code{Seurat} object and values
are columns of \code{var} in \code{adata}. See below for details.}

\item{reduction_mapping}{A named vector where names are names of \code{Embeddings}
in the resulting \code{Seurat} object and values are keys of \code{obsm} in \code{adata}.
Alternatively, a named list where names are names of \code{Embeddings} in the
resulting \code{Seurat} object and values are vectors with the items \code{"key"},
\code{"embeddings"} and (optionally) \code{"loadings"}. See below for details.}

\item{graph_mapping}{A named vector where names are names of \code{Graphs} in the
resulting \code{Seurat} object and values are keys of \code{obsp} in \code{adata}. See
below for details.}

\item{misc_mapping}{A named vector where names are names of \code{Misc} in the
resulting \code{Seurat} object and values are keys of \code{uns} in \code{adata}. See
below for details.}
}
\value{
A \code{Seurat} object containing the requested data from \code{adata}
}
\description{
Convert an \code{AnnData} object to a \code{Seurat} object
}
\details{
\subsection{Mapping arguments}{

All mapping arguments expect a named character vector where names are the
names of the slot in the \code{Seurat} object and values are the keys of the
corresponding slot of \code{adata}. If \code{TRUE}, the conversion function will guess
which items to copy as described in the conversion table below. In most
cases, the default is to copy all items using the same names except where the
correspondence between objects is unclear. The \code{reduction_mapping} argument
can also accept a more complex list format, see below for details. To avoid
copying anything to a slot, set the mapping argument to \code{FALSE}. Empty
mapping arguments (\code{NULL}, \code{c()}, \code{list()}) will be treated as \code{FALSE} with
a warning. If an unnamed vector is provided, the values will be used as
names.
\subsection{Examples:}{
\itemize{
\item \code{TRUE} will guess which items to copy as described in the conversion
table
\item \code{c(seurat_item = "adata_item")} will copy \code{adata_item} from the slot in
\code{adata} to \code{seurat_item} in the corresponding slot of the new \code{Seurat}
object
\item \code{FALSE} will avoid copying anything to the slot
\item \code{c("adata_item")} is equivalent to \code{c(adata_item = "adata_item")}
}
}

}

\subsection{Conversion table}{\tabular{llll}{
   \strong{From \code{AnnData}} \tab \strong{To \code{Seurat}} \tab \strong{Example mapping argument} \tab \strong{Default} \cr
   \code{adata$X} \tab \code{Layers(seurat)} \tab \code{x_mapping = "counts"} \emph{OR} \code{layers_mapping = c(counts = NA)} \tab The data in \code{adata$X} is copied to a layer named \code{X} \cr
   \code{adata$layers} \tab \code{Layers(seurat)} \tab \code{layers_mapping = c(counts = "counts")} \tab All items are copied by name \cr
   \code{adata$obs} \tab \code{seurat[[]]} \tab \code{object_metadata_mapping = c(n_counts = "n_counts", cell_type = "CellType")} \tab All columns are copied by name \cr
   \code{adata$var} \tab \code{seurat[[assay_name]][[]]} \tab \code{assay_metadata_mapping = c(n_cells = "n_cells", pct_zero = "PctZero")} \tab All columns are copied by name \cr
   \code{adata$obsm} \tab \code{Reductions(seurat)} \tab \code{reduction_mapping = c(pca = "X_pca")} \strong{OR} \code{reduction_mapping = list(pca = c(key = "PC_", embeddings = "X_pca", loadings = "PCs"))} \tab All items that can be coerced to a numeric matrix are copied by name without loadings except for \code{"X_pca"} for which loadings are added from \code{"PCs"} \cr
   \code{adata$obsp} \tab \code{Graphs(seurat)} \tab \code{graph_mapping = c(nn = "connectivities")} \tab All items are copied by name \cr
   \code{adata$varp} \tab \emph{NA} \tab \emph{NA} \tab There is no corresponding slot for \code{varp} \cr
   \code{adata$uns} \tab \code{Misc(seurat)} \tab \code{misc_mapping = c(project_metadata = "metadata")} \tab All items are copied by name \cr
}

}

\subsection{The \code{reduction_mapping} argument}{

For the simpler named vector format, the names should be the names of
\code{Embeddings} in the resulting \code{Seurat} object, and the values
should be the keys of \code{obsm} in \code{adata}. A key will created from the \code{obsm}
key.

For more advanced mapping, use the list format where each item is a vector
with the following names defining arguments to
\code{\link[SeuratObject:CreateDimReducObject]{SeuratObject::CreateDimReducObject()}}:
\itemize{
\item \code{key}: the key of the resulting \code{\link[SeuratObject:DimReduc-class]{SeuratObject::DimReduc}} object, passed
to the \code{key} argument after processing with \code{\link[SeuratObject:Key]{SeuratObject::Key()}}
\item \code{embeddings}: a key of the \code{obsm} slot in \code{adata},
\code{adata$obsm[[embeddings]]} is passed to the \code{embeddings} argument
\item \code{loadings}: a key of the \code{varm} slot in \code{adata} (optional),
\code{adata$varm[[loadings]]} is passed to the \code{loadings} argument
}
}

\subsection{The \code{x_mapping} and \code{layers_mapping} arguments}{

In order to specify where the data in \code{adata$X} will be stored in the
\code{Layers(seurat)} slot of the resulting object, you can use either the \code{x_mapping}
argument or the \code{layers_mapping} argument.
If you use \code{x_mapping}, it should be a string specifying the name of the layer
in \code{Layers(seurat)} where the data in \code{adata$X} will be stored.
If you use \code{layers_mapping}, it should be a named vector where names are names
of \code{Layers(seurat)} and values are keys of \code{layers} in \code{adata}.
In order to indicate the \code{adata$X} slot, you use \code{NA} as the value in the vector.
The name you provide for \code{x_mapping} may not be a name in \code{layers_mapping}.
}
}
\examples{
\dontshow{if (rlang::is_installed("Seurat")) withAutoprint(\{ # examplesIf}
  ad <- AnnData(
    X = matrix(1:5, 3L, 5L),
    obs = data.frame(row.names = LETTERS[1:3], cell = 1:3),
    var = data.frame(row.names = letters[1:5], gene = 1:5)
  )

  # Default usage
  seurat <- ad$as_Seurat(
    assay_name = "RNA",
    x_mapping = "counts",
    layers_mapping = TRUE,
    object_metadata_mapping = TRUE,
    assay_metadata_mapping = TRUE,
    reduction_mapping = TRUE,
    graph_mapping = TRUE,
    misc_mapping = TRUE
  )
\dontshow{\}) # examplesIf}
}
\seealso{
Other object converters: 
\code{\link{as_AnnData}()},
\code{\link{as_HDF5AnnData}()},
\code{\link{as_InMemoryAnnData}()},
\code{\link{as_ReticulateAnnData}()},
\code{\link{as_SingleCellExperiment}()},
\code{\link{reticulate-helpers}}
}
\concept{object converters}
