#' Read a VCF object from disk
#'
#' Read a \linkS4class{VCF} object from its on-disk representation.
#'
#' @param path String containing a path to a directory, usually generated by the \code{\link{saveObject}} method for \linkS4class{VCF} object.s
#' @param metadata Named list of metadata for this object, see \code{\link{readObjectFile}} for details.
#' @param ... Further arguments passed to internal \code{\link{altReadObject}} calls.
#'
#' @return A \linkS4class{VCF} object. 
#'
#' @seealso
#' \code{\link{saveObject,VCF-method}}, to save VCF objects to disk.
#'
#' @author Aaron Lun
#'
#' @examples
#' fl <- system.file("extdata", "structural.vcf", package="VariantAnnotation")
#' vcf <- readVcf(fl)
#'
#' tmp <- tempfile()
#' saveObject(vcf, tmp)
#' readObject(tmp)
#' 
#' @export
#' @aliases
#' loadVCF
#' loadVCFHeader
#' @import alabaster.base VariantAnnotation
readVCF <- function(path, metadata, ...) {
    # Note that this doesn't work with regular Gzipped files right now, only
    # BGZF-compressed files, see Bioconductor/VariantAnnotation#32.
    out <- readVcf(file.path(path, "file.vcf.gz"))
    if (isTRUE(metadata$vcf_experiment$expanded)) {
        out <- expand(out)
    }
    out
}

########################
#### OLD STUFF HERE ####
########################

#' @export
#' @importFrom alabaster.se loadSummarizedExperiment
loadVCF <- function(vcf.info, project) {
    out <- loadSummarizedExperiment(vcf.info, project)

    fixed.meta <- acquireMetadata(project, vcf.info$vcf_experiment$fixed$resource$path)
    fixed.df <- .loadObject(fixed.meta, project)

    info.meta <- acquireMetadata(project, vcf.info$vcf_experiment$info$resource$path)
    info.df <- .loadObject(info.meta, project)

    header.meta <- acquireMetadata(project, vcf.info$vcf_experiment$header$resource$path)
    metadata(out)$header <- .altLoadVCFHeader()(header.meta, project)

    if (!is(out, "RangedSummarizedExperiment")) {
        out <- as(out, "RangedSummarizedExperiment")
    }

    if (vcf.info$vcf_experiment$expanded) {
        new("ExpandedVCF", out, fixed=fixed.df, info=info.df)
    } else {
        new("CollapsedVCF", out, fixed=fixed.df, info=info.df)
    }
}
