\name{baselineCorrection}
\alias{baselineCorrection}
\title{Baseline correction - Chang's method}
\description{
    Function for baseline correction of GC-MS chromatograms using Chang's method
    described below.
}
\usage{
    baselineCorrection(peaks, threshold = 0.5, alpha = 0.95, bfraction = 0.2,
           segments = 100, signalWindow = 10, method = "linear")
}
\arguments{
  \item{peaks}{Either a matrix object of spectra peak intensities to be baseline corrected,
  where the rows are retention times and columns are mass traces; or, a named list containing
  an element called \code{"Peaks"} which such matrix. The list can be generated by
  \code{\link{peakCDFextraction}}}
  \item{threshold}{A numeric value between 0 and 1. A value of one sets the baseline
  above the noise, 0.5 in the middle of the noise and 0 below the noise.}
  \item{alpha}{The alpha parameter of the high pass filter.}
  \item{bfraction}{The percentage of the fragments with the lowest intensities of
  the filtered signal that are assumed to be baseline signal.}
  \item{segments}{The number of segments in which the filtered signal is divided.}
  \item{signalWindow}{The window size (number of points) used in the signal windowing step.}
  \item{method}{The method used to approximate the baseline. \code{"linear"} (default) uses
  linear interpolation. \code{"spline"} fits a cubic smoothing spline (warning: really slow). }
}
\details{
The baseline correction algorithm is based on the work of Chang et al, and it works as
follows. For every mass trace, i.e., columns of matrix \code{peaks}, the signal intensity is filtered
by a first high pass filter: \deqn{y_i = \alpha (y_{i-1} + x_i - x_{i-1})} The
filtered signal is divided into evenly spaced segments (\code{segments})
and the standard deviation of each segment is calculated. A percentage (\code{bfraction})
of the segments with the lowest values are assumed to be baseline signal and the
standard deviation (\eqn{\sigma}{stdn}) of the points within those segments is calculated.

Once \eqn{\sigma}{stdn} has been determined, the points with absolute filtered values larger than
\eqn{2\sigma}{2 * stdn} are considered signal. After that, the signal windowing step takes
every one of the points found to be signal as the center of a signal window (\code{signalWindow})
and marks the points within that window as signal. The remaining points are now considered
to be noise.

The baseline signal is obtained by either using linear interpolation (default)
or fitting a cubic smoothing spline taking only
the noise. The baseline can be shifted up or down by using the parameter
\code{threshold}, which is done by the formula:
\deqn{B' = B + 4\sigma(t - 0.5)}{B' = B + 4 * stdn (t - 0.5)}
where
\eqn{B} is the fitted spline, \eqn{\sigma}{stdn} the standard deviation of the noise,
and \eqn{t} is the \code{threshold} between 0 and 1. Finally, the corrected signal
is calculated by subtracting \eqn{B'} to the original signal.

}
\value{
The output depends on whether the input \code{peaks} is a matrix or a list. If it is a matrix,
then the function returns a matrix of the same dimensions with the baseline corrected intensities.
If instead \code{peaks} is a list, then the element called \code{"Peaks"} will hold the
output.
}

\note{
This function is intended to be run internally, but it is exported for advanced users.
}
\references{
David Chang, Cory D. Banack and Sirish L. Shah, Robust baseline correction algorithm
for signal dense NMR spectra. \emph{Journal of Magnetic Resonance 187 (2007) 288-292}
}
\author{Alvaro Cuadros-Inostroza}
\seealso{ \code{\link{RIcorrect}}, \code{\link{baseline} }}
\examples{
  # get a random sample CDF from TargetSearchData
  require(TargetSearchData)
  cdffile <- sample(tsd_cdffiles(), 1)
  pdata <- peakCDFextraction(cdffile)

  # restrict mass range to reduce computing time (not needed for
  # actual data)
  pdata$Peaks <- pdata$Peaks[, 1:10] ; pdata$massRange <- c(85, 94)

  # make a fake baseline as constant + noise (the CDF files have been
  # already baseline corrected by the vendor software).
  nscans <- length(pdata$Time)
  noise <- as.integer(1000 + rnorm(nscans, sd=5))
  pdata$Peaks <- pdata$Peaks + noise

  # change parameters and see how the results change
  pdata1 <- baselineCorrection(pdata)
  pdata2 <- baselineCorrection(pdata, threshold = 1, alpha = 0.97)

  # pick random trace k
  k <- 6
  m <- cbind(pdata$Peaks[, k] - noise, pdata1$Peaks[, k], pdata2$Peaks[, k])
  matplot(pdata$Time, m, type='l', lty=1, xlab='time', ylab='intensity')
  legend('topleft', c('original', 'base correct 1', 'base correct 2'),
         col=1:3, lty=1, lwd=1)
}
% vim: ts=2 sw=2 et
