% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Spectra-precursorPurity.R
\name{precursorPurity}
\alias{precursorPurity}
\title{Calculating Precursor Purity for MS2 spectra}
\usage{
precursorPurity(
  object,
  tolerance = 0.05,
  ppm = 0,
  useReportedIsolationWindow = FALSE,
  BPPARAM = SerialParam()
)
}
\arguments{
\item{object}{\code{\link[=Spectra]{Spectra()}} object with LC-MS/MS data.}

\item{tolerance}{\code{numeric(1)} defining an absolute value (in Da) to be used
to define the isolation window. For the precursor purity calculation of
an MS2 spectrum, all MS1 peaks from the previous MS1 scan with an m/z
between the fragment spectrum's precursorMz +/- (tolerance +
ppm(precursorMz, ppm)) are considered.}

\item{ppm}{\code{numeric(1)} defining the m/z dependent acceptable difference in
m/z. See documentation of parameter \code{tolerance} for more information.}

\item{useReportedIsolationWindow}{\code{logical(1)} whether the reported
isolation window, defined by spectra variables \code{isolationWindowLowerMz}
and \code{isolationWindowUpperMz} in the input \link{Spectra} object, should be
used instead of calculating the isolation window from the reported
precursor m/z and parameters \code{tolerance} and \code{ppm}. Only few
manufacturers report the isolation window with the spectra variables
\code{isolationWindowLowerMz} and \code{isolationWindowTargetMz}, thus the default
for this parameter is \code{FALSE}.}

\item{BPPARAM}{parallel processing setup. Defaults to
\code{BPPARAM = SerialParam()}. See \code{\link[BiocParallel:SerialParam-class]{BiocParallel::SerialParam()}} for
more information.}
}
\value{
\code{numeric} vector of length equal to the number of spectra in
\code{object},  with values representing the calculated precursor purity
for each spectrum. For MS1 spectra, \code{NA_real_} is returned. For MS2
spectra, the purity is defined as the proportion of maximum signal to
the total ion current within the isolation  window that is attributable
to the selected precursor ion. If no matching  MS1 scan is found or
the precursor peak is missing, \code{NA_real_} is returned.
}
\description{
MS instruments generally collect precursor ions in a discrete \emph{m/z}
\emph{isolation window} before fragmenting them and recording the respective
fragment (MS2) spectrum. Ideally, only a single ion species is fragmented,
depending also on the size of the isolation window, different ions (with
slightly different \emph{m/z}) might be fragmented. The resulting MS2 spectrum
might thus contain fragments from different ions and hence be less \emph{pure}.

The \code{precursorPurity()} function calculates the \strong{precursor purity} of MS2
(fragment) spectra expressed as the ratio between the itensity of the highest
signal in the isolation window to the sum of intensities of all MS1 peaks in
the isolation window. This is similar to the calculation performed in the
\href{https://www.bioconductor.org/packages/release/bioc/html/msPurity.html}{\emph{msPurity}}
Bioconductor package.

The peak intensities within the isolation window is extracted from the last
MS1 spectrum before the respective MS2 spectrum. The spectra are thus
expected to be ordered by retention time. For the isolation window either the
isolation window reported in the \code{Spectra} object is used, or it is
calculated based on the MS2 spectra's precursor m/z. By default, the
isolation window is calculated based on the precursor m/z and parameters
\code{tolerance} and \code{ppm}: precursorMz +/- (\code{tolerance} +
\code{ppm(precursorMz, ppm)}). If the actually used precursor isolation window is
defined and available in the \code{Spectra} object, it can be used instead by
setting \code{useReportedIsolationWindow = TRUE} (default is
\code{useReportedIsolationWindow = FALSE}). Note that parameters \code{tolerance}
and \code{ppm} are ignored for \code{useReportedIsolationWindow = TRUE}.
}
\note{
This approach is applicable only when fragment spectra are obtained through
data-dependent acquisition (DDA), as it assumes that the peak with the
highest intensity within the given isolation m/z window (from the previous
MS1 spectrum) corresponds to the precursor ion.

The spectra in \code{object} have to be ordered by their retention time.
}
\examples{

## Load a test DDA file
library(msdata)
fl <- system.file("TripleTOF-SWATH", "PestMix1_DDA.mzML",
                 package = "msdata")
sps_dda <- Spectra(fl)

## Define the isolation window based on the MS2 spectra's precursor m/z
## and parameter `tolerance`: isolation window with size 1Da:
pp <- precursorPurity(sps_dda, tolerance = 0.5)

## values for MS1 spectra are NA
head(pp[msLevel(sps_dda) == 1])

head(pp[msLevel(sps_dda) == 2])

## Use the reported isolation window (if defined in the `Spectra`):
filterMsLevel(sps_dda, 2L) |>
    isolationWindowLowerMz() |>
    head()
filterMsLevel(sps_dda, 2L) |>
    isolationWindowUpperMz() |>
    head()

pp_2 <- precursorPurity(sps_dda, useReportedIsolationWindow = TRUE)

head(pp_2[msLevel(sps_dda) == 2])
}
\seealso{
\code{\link[=addProcessing]{addProcessing()}} for other data analysis and manipulation functions.
}
\author{
Ahlam Mentag, Johannes Rainer
}
