% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/transforms.R
\name{setaBalance}
\alias{setaBalance}
\title{User-defined balance transform (geometric-mean log-ratio)}
\usage{
setaBalance(
  counts,
  balances,
  taxonomyDF = NULL,
  taxonomy_col = NULL,
  normalize_to_parent = FALSE,
  pseudocount = 1
)
}
\arguments{
\item{counts}{Numeric matrix with rows = samples and columns = taxa.}

\item{balances}{A single balance (list with `num`, `denom`) **or**
a named list of such lists for multiple balances.}

\item{taxonomyDF}{Optional. A data frame from [setaTaxonomyDF()] used to
expand higher-level labels into their descendant leaves.}

\item{taxonomy_col}{Character. Column in `taxonomyDF` whose values should
match any higher-level labels given in `balances`.}

\item{normalize_to_parent}{Logical (default `FALSE`). If `TRUE`, each sample
is re-closed to the sub-composition formed by the union of \code{num} and \code{denom} before
taking the log-ratio, i.e., the balance is within the parent total.}

\item{pseudocount}{Numeric. Value added to every count to avoid
`log(0)`. Default `1`.}
}
\value{
A list with
\describe{
  \item{method}{\code{"balance"}.}
  \item{counts}{Matrix with dimensions samples \eqn{\times} balances. Column names are the
    balance names (or \code{"Balance1"} if unnamed).}
}
}
\description{
`setaBalance()` computes *one or more* biologically meaningful balances
(log-ratios) from a count matrix. Each balance is defined by two
groups of taxa: **numerator** (`num`) and **denominator** (`denom`).
Groups may be given as leaf names, higher-level labels (resolved through a
`taxonomyDF`), or column indices. The resulting balance will be positive
if weighted in the numerator direction, and negative toward the denominator.
}
\details{
For every balance and every sample the function returns
\deqn{\log \big( \mathrm{GM}(\mathrm{num}) / \mathrm{GM}(\mathrm{denom}) \big),}
where \eqn{\mathrm{GM}(\cdot)} is the geometric mean of the (pseudocount-adjusted) counts in
the respective group.
}
\examples{
# Toy metadata & taxonomy table (from setaTaxonomyDF documentation)
meta <- data.frame(
  bc         = paste0("cell", 1:6),
  fine_type  = c("AT1","AT2","AT1","Fib1","Fib1","AT2"),
  mid_type   = c("Alv","Alv","Alv","Fib","Fib","Alv"),
  broad_type = c("Epi","Epi","Epi","Stroma","Stroma","Epi")
)
taxDF <- setaTaxonomyDF(meta,
  resolution_cols = c("broad_type","mid_type","fine_type"))

# Fake counts (2 samples x n_taxa leaves)
set.seed(687)
cnt <- matrix(rpois(2 * 3, 10), nrow = 2)
colnames(cnt) <- rownames(taxDF)

# (a) One balance: Epi vs Stroma (broad_type level)
bal1 <- list(num = "Epi", denom = "Stroma")
out1 <- setaBalance(cnt, bal1,
  taxonomyDF = taxDF, taxonomy_col = "broad_type")
out1$counts

# (b) Two balances in one call
bals <- list(
  epi_vs_stroma = list(num = "Epi", denom = "Stroma"),
  AT1_vs_AT2    = list(num = "AT1", denom = "AT2")
)
out2 <- setaBalance(cnt, bals,
  taxonomyDF = taxDF, taxonomy_col = "fine_type")
out2$counts
}
