% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get_metrics.R, R/bootCI.R
\name{getMetrics}
\alias{getMetrics}
\alias{bootCI}
\alias{bootGloscope}
\title{Calculate metrics on separation in distance matrix due to grouping variable}
\usage{
getMetrics(
  dist_mat,
  metadata_df,
  metrics = c("anosim", "adonis2", "silhouette"),
  sample_id,
  group_vars,
  checkData = TRUE,
  permuteTest = FALSE,
  permutations = 100
)

bootCI(
  dist_mat,
  metadata_df,
  metrics = "anosim",
  sample_id,
  group_vars,
  R = 1000,
  ci_type = c("perc", "norm", "basic", "stud", "bca"),
  ci_conf = 0.95,
  ...
)

bootGloscope(
  dist_mat,
  metadata_df,
  metrics = "anosim",
  sample_id,
  group_vars,
  R = 1000,
  ...
)
}
\arguments{
\item{dist_mat}{The divergence matrix output of `gloscope()`. Should be a
symmetric, square matrix. For `bootCI` the argument can be a list
of distance matrices.}

\item{metadata_df}{A data frame contains each sample's metadata. Note this is
NOT at the cell-level, and should have the same number of rows as dist_mat.}

\item{metrics}{vector of statistics to calculate. For `bootstrap_gloscope`
must be single value.}

\item{sample_id}{The column name or index in metadata_df that contains the
sample ID. This is for ensuring alignment between the dist_mat and the
metadata_df. The rownames of dist_mat are expected to match the sample_id
values.}

\item{group_vars}{vector of names of grouping variables in metadata_df for
which to calculate metrics. For `bootstrap_gloscope` must be single value.}

\item{checkData}{Whether to check whether dist_mat, metadata_df, and
sample_id match, for example in terms of dimensions and rownames. Mainly
used internally.}

\item{permuteTest}{whether to run permutation tests on each of the metrics}

\item{permutations}{if `permuteTest=TRUE`, an integer value defines the
number of permutations. Can also except output of
\code{\link[permute]{how}} for more fine control of the permutation
mechanisms.}

\item{R}{number of bootstrap replicates. See \code{\link[boot]{boot}}.}

\item{ci_type}{Single character value. The type of confidence interval to
compute. Passed to argument `type` in \code{\link[boot]{boot.ci}}}

\item{ci_conf}{Scalar value between 0 and 1. The confidence level requested.
Passed to argument `conf` in \code{\link[boot]{boot.ci}}.}

\item{...}{arguments passed to \code{\link[boot]{boot}}}
}
\value{
`getMetrics` creates a data frame containing the statistic for each
  combination of metric and grouping variable with columns
\itemize{
  \item metric 
  \item grouping
  \item statistic
  \item pval (if `permuteTest=TRUE`)
}

`bootCI` creates a data frame containing the statistic for
  each combination of metric and grouping variable with columns with the
  upper and lower bounds of the requested confidence intervals
\itemize{
  \item metric 
  \item grouping
  \item statistic
  \item lower
  \item upper
}

`bootGloscope` returns an object of class `boot` created by \code{\link[boot]{boot}}.
}
\description{
These functions are wrappers for calculating common metrics for
  the amount of separation in a distance matrix due to a grouping (factor)
  variable and creating bootstrap confidence intervals and permutation tests.
}
\details{
The function `getMetrics` is a simple wrapper for calculating statistics that
  summarize the difference between distances within and between groupings. If
  the variable defined by group_var does not have at least two groupings, the
  function will return a NA.

The options "anosim" and "adonis2" are wrappers to the functions of
  that name in the package `vegan`; we have turned off the permutation
  testing option of those functions. The functions in `vegan` have greater
  capability, and in particular \code{\link[vegan]{adonis2}} has capability
  to handle more complicated testing paradigms than a simple grouping factor.
  Permutation tests for these statistics are handled by the functions of
  `vegan`.

The option "silhouette" calls the function of that name from the
  package `cluster` and calculates the silhouette width of each group and
  then averages them across groups. The permutation test is coded making use
  of the package `permute`, similar to `vegan`, so that control of the
  permutation mechanism is possible in the same way.

`bootCI` is a wrapper function to
  \code{\link[boot]{boot.ci}}. `boot.ci` can be called directly on the output
  of `bootGloscope`. The main advantage of `bootCI` is to calculate
  bootstrap CI over multiple choices of metrics, variables, and/or distance
  matrices. Unlike `boot.ci`, `bootCI` does not allow different
  choices of confidence interval types or levels, so `ci_type` and `ci_level`
  must be of length 1. For this kind of multiplicity, call `boot.ci` directly
  on the output of `bootGloscope`.

The function `bootGloscope` is a wrapper to the
  \code{\link[boot]{boot}} function for creating bootstraps of one of the
  metrics calculated by `getMetrics`. Most users will probably prefer `bootCI`
}
\examples{
data(example_SCE_small)
sample_ids <- SingleCellExperiment::colData(example_SCE_small)$sample_id
# Run gloscope on first 10 PCA embeddings
# We use 'KNN' option for speed ('GMM' is slightly slower)
pca_embeddings <- SingleCellExperiment::reducedDim(example_SCE_small,"PCA")
pca_embeddings_subset <- pca_embeddings[,seq_len(10)] # select the first 10 PCs
dist_result <- gloscope(pca_embeddings_subset, sample_ids,
   dens="KNN",
   BPPARAM = BiocParallel::SerialParam(RNGseed=2))
# make a per-sample metadata
sample_metadata <- as.data.frame(unique(SingleCellExperiment::colData(example_SCE_small)[,c(1,2)]))
# make another variable
sample_metadata$grouping<-c(rep(c("A","B"),each=2),"A")
getMetrics(dist_result,metadata_df=sample_metadata, sample_id="sample_id", 
  group_vars="phenotype")
# run permutation tests:
getMetrics(dist_result,metadata_df=sample_metadata, sample_id="sample_id", 
  group_vars=c("phenotype","grouping"), permuteTest=TRUE)
  
# calculate many bootstraps -- for speed up we set R ridiculously low
manyboot<-bootCI(list("Distance 1"=dist_result,"Another distance"=dist_result),
  sample_metadata,"sample_id",
  metrics=c("anosim","silhouette"),group_vars=c("phenotype","grouping"),R=20)

# single bootstrap of anosim
bootout<-bootGloscope(dist_result,sample_metadata,"sample_id",
  metric="anosim",group_var="phenotype")
#work with the boot object using functions in boot package:
library(boot)
print(bootout)
boot.ci(bootout)

}
\seealso{
\code{\link[vegan]{anosim}}, \code{\link[vegan]{adonis2}}, 
 \code{\link[cluster]{silhouette}}, \code{\link[permute]{how}}

\code{\link[boot]{boot.ci}}

\code{\link[boot]{boot}}
}
