#' GR values
#'
#' Given a SummarizedExperiment object created by \code{\link{GRfit}},
#' this function returns a table of the original data (in the form of 
#' "Case A") with columns for GR values and relative cell counts.
#'
#' @param fitData a SummarizedExperiment object, generated by the GRfit 
#' function.
#'
#' @return A table with GR values and relative cell counts
#' @author Nicholas Clark
#' @details
#' Given a SummarizedExperiment object created by \code{\link{GRfit}},
#' this function returns a table of the original data (in the form of 
#' "Case A") with columns for GR values and relative cell counts. This is 
#' simply a convenient accessor function, equivalent to 
#' \code{as.data.frame(metadata(fitData)[[1]])}.
#' @seealso To create the object needed for this function, see
#' \code{\link{GRfit}}.
#' @examples
#' # Load Case A (example 1) input
#' data("inputCaseA")
#' head(inputCaseA)
#' # Run GRfit function with case = "A"
#' output1 = GRfit(inputData = inputCaseA,
#' groupingVariables = c('cell_line','agent'))
#' GRvalues = GRgetValues(output1)
#' head(GRvalues)
#' \dontrun{View(GRvalues)}
#' @export

GRgetValues <- function(fitData) {
  return(as.data.frame(metadata(fitData)[[1]]))
}


#' GR metrics
#'
#' Given a SummarizedExperiment object created by \code{\link{GRfit}},
#' this function returns a table of GR metrics (as well as traditional 
#' metrics) for each experiment in the dataset.
#'
#' @param fitData a SummarizedExperiment object, generated by the GRfit 
#' function.
#'
#' @return A table with GR metrics and goodness of fit measures
#' @author Nicholas Clark
#' @details
#' Given a SummarizedExperiment object created by \code{\link{GRfit}},
#' this function returns a table of GR metrics and traditional metrics 
#' along with goodness of fit measures. It also identifies each fit 
#' as flat or sigmoidal. This is simply a convenient accessor function, 
#' equivalent to \code{cbind(as.data.frame(colData(fitData)), 
#' as.data.frame(t(assay(fitData))))}.
#' @seealso To create the object needed for this function, see
#' \code{\link{GRfit}}.
#' @examples
#' # Load Case A (example 1) input
#' data("inputCaseA")
#' head(inputCaseA)
#' # Run GRfit function with case = "A"
#' output1 = GRfit(inputData = inputCaseA,
#' groupingVariables = c('cell_line','agent'))
#' metrics = GRgetMetrics(output1)
#' \dontrun{View(metrics)}
#' head(metrics)
#' @export

GRgetMetrics <- function(fitData) {
  return(cbind(as.data.frame(SummarizedExperiment::colData(fitData)), 
               as.data.frame(t(SummarizedExperiment::assay(fitData)))))
}

#' Grouping Variables
#'
#' Given a SummarizedExperiment object created by \code{\link{GRfit}},
#' this function returns a vector of the grouping variables used to create 
#' the object.
#'
#' @param fitData a SummarizedExperiment object, generated by the GRfit 
#' function.
#'
#' @return A vector of grouping variables
#' @author Nicholas Clark
#' @details
#' Given a SummarizedExperiment object created by \code{\link{GRfit}},
#' this function returns a vector of the grouping variables used to create the 
#' object. These are the variables in the dataset that are not averaged over. 
#' This is simply a convenient accessor function, equivalent to 
#' \code{metadata(fitData)[[2]]}.
#' @seealso To create the object needed for this function, see
#' \code{\link{GRfit}}.
#' @examples
#' # Load Case A (example 1) input
#' data("inputCaseA")
#' head(inputCaseA)
#' # Run GRfit function with case = "A"
#' output1 = GRfit(inputData = inputCaseA,
#' groupingVariables = c('cell_line','agent'))
#' groupVars = GRgetGroupVars(output1)
#' groupVars
#' @export

GRgetGroupVars <- function(fitData) {
  return(metadata(fitData)[[2]])
}

#' GR metrics definitions
#'
#' Given a SummarizedExperiment object created by \code{\link{GRfit}},
#' this function returns a table with the definition of each GR metric and 
#' traditional metric that is calculated.
#'
#' @param fitData a SummarizedExperiment object, generated by the GRfit 
#' function.
#'
#' @return A table of definitions of GR metrics and traditional metrics
#' @author Nicholas Clark
#' @details
#' Given a SummarizedExperiment object created by \code{\link{GRfit}},
#' this function returns a table containing the definition of each GR metric,
#' traditional metric, and goodness of fit measure in the object. 
#' This is simply a convenient accessor function, equivalent to 
#' \code{rowData(fitData)}.
#' @seealso To create the object needed for this function, see
#' \code{\link{GRfit}}.
#' @examples
#' # Load Case A (example 1) input
#' data("inputCaseA")
#' head(inputCaseA)
#' # Run GRfit function with case = "A"
#' output1 = GRfit(inputData = inputCaseA,
#' groupingVariables = c('cell_line','agent'))
#' defs = GRgetDefs(output1)
#' # See all definitions (some will be truncated)
#' \dontrun{View(defs)}
#' # See the first six descriptions
#' head(defs)
#' # Look at a specific definition
#' defs[15,]
#' @export

GRgetDefs <- function(fitData) {
  return(SummarizedExperiment::rowData(fitData))
}