#' tomaftools
#' @description Transform a CaMutQC maf object to a maftools maf object.
#'
#' @param maf An MAF data frame, generated by \code{\link{vcfToMAF}} function.
#' @param clinicalData Clinical data associated with each 
#' # sample/Tumor_Sample_Barcode in MAF. Could be a text file or a data.frame. 
#' Default NULL. Inherited from maftools.
#' @param useAll logical. Whether to use all variants irrespective of values in 
#' Mutation_Status. Defaults to TRUE. If FALSE, only uses with values Somatic.
#' Inherited from maftools.
#' @param rmFlags Default FALSE. Can be TRUE or an integer. 
#' If TRUE, removes all the top 20 FLAG genes. 
#' If integer, remove top n FLAG genes. Inherited from maftools.
#' @param gisticAllLesionsFile All Lesions file generated by gistic. 
#' e.g; all_lesions.conf_XX.txt, where XX is the confidence level. Default NULL.
#' Inherited from maftools.
#' @param gisticAmpGenesFile Amplification Genes file generated by gistic. 
#' e.g; amp_genes.conf_XX.txt, where XX is the confidence level. Default NULL.
#' Inherited from maftools.
#' @param gisticDelGenesFile Deletion Genes file generated by gistic. 
#' e.g; del_genes.conf_XX.txt, where XX is the confidence level. Default NULL.
#' Inherited from maftools.
#' @param gisticScoresFile scores.gistic file generated by gistic. Default NULL
#' Inherited from maftools.
#' @param cnLevel level of CN changes to use. Can be 'all', 'deep' or 'shallow'.
#' Default uses all i.e, genes with both 'shallow' or 'deep' CN changes.
#' Inherited from maftools.
#' @param cnTable Custom copynumber data if gistic results are not available. 
#' Input file or a data.frame should contain three columns in aforementioned 
#' order with gene name, Sample name and copy number status 
#' (either 'Amp' or 'Del'). Default NULL. Inherited from maftools.
#' @param isTCGA Is input MAF file from TCGA source. 
#' If TRUE uses only first 12 characters from Tumor_Sample_Barcode.
#' Inherited from maftools.
#' @param removeDuplicatedVariants removes repeated variants 
#' in a particuar sample, mapped to multiple transcripts of same Gene. 
#' See Description. Default TRUE. Inherited from maftools.
#' @param vc_nonSyn NULL. Provide manual list of variant classifications to be 
#' considered as non-synonymous. Rest will be considered as silent variants. 
#' Default uses Variant Classifications with High/Moderate variant consequences. 
#' Inherited from maftools.
#' @param verbose TRUE logical. Default to be talkative and prints summary.
#' Inherited from maftools.
#' 
#' @return An maf object that can be recognized by maftools.
#' 
#' @importFrom maftools read.maf
#' @importFrom methods is
#' 
#' @export tomaftools
#' @examples
#' maf_CaMutQC <- vcfToMAF(system.file("extdata/Multi-caller/",
#' package="CaMutQC"), multiVCF=TRUE)
#' maf_maftools <- tomaftools(maf_CaMutQC)

tomaftools <- function(maf, clinicalData = NULL, rmFlags = FALSE, 
                       removeDuplicatedVariants = TRUE, useAll = TRUE, 
                       gisticAllLesionsFile = NULL, gisticAmpGenesFile = NULL,
                       gisticDelGenesFile = NULL, gisticScoresFile = NULL, 
                       cnLevel = 'all', cnTable = NULL, isTCGA = FALSE, 
                       vc_nonSyn = NULL, verbose = TRUE){
    # check user input
    if (!(is(maf, "data.frame"))) {
        stop("maf input should be a data frame, did you get it from vcfToMAF function?")
    }
    
    message("Transforming to maftools maf...")
    # write to a temporary file and read it using MesKit immediately
    write.table(maf, "./tmp_maf_for_maftools.maf", sep="\t", 
                quote=FALSE, row.names=FALSE)
    mafFinal <- maftools::read.maf("./tmp_maf_for_maftools.maf", 
                  clinicalData=clinicalData, rmFlags=rmFlags, 
                  removeDuplicatedVariants=removeDuplicatedVariants, 
                  useAll=useAll, gisticAllLesionsFile=gisticAllLesionsFile, 
                  gisticAmpGenesFile=gisticAmpGenesFile,
                  gisticDelGenesFile=gisticDelGenesFile, 
                  gisticScoresFile=gisticScoresFile, 
                  cnLevel=cnLevel, cnTable=cnTable, isTCGA=isTCGA, 
                  vc_nonSyn=vc_nonSyn, verbose=verbose)
    # remove temporary file immediately
    unlink("./tmp_maf_for_maftools.maf", recursive=FALSE, force=FALSE)
    # return maftools object maf
    return(mafFinal)
}

