% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/CTSSs.R
\name{quantifyCTSSs}
\alias{quantifyCTSSs}
\alias{quantifyCTSSs,BigWigFileList,BigWigFileList-method}
\alias{quantifyCTSSs,character,character-method}
\title{Quantify CAGE Transcriptions Start Sites (CTSSs)}
\usage{
quantifyCTSSs(plusStrand, minusStrand, design = NULL, genome = NULL, ...)

\S4method{quantifyCTSSs}{BigWigFileList,BigWigFileList}(
  plusStrand,
  minusStrand,
  design = NULL,
  genome = NULL,
  nTiles = 1L
)

\S4method{quantifyCTSSs}{character,character}(plusStrand, minusStrand, design = NULL, genome = NULL)
}
\arguments{
\item{plusStrand}{BigWigFileList or character: BigWig/bedGraph files with plus-strand CTSS data.}

\item{minusStrand}{BigWigFileList or character: BigWig/bedGraph files with minus-strand CTSS data.}

\item{design}{DataFrame or data.frame: Additional information on samples which will be added to the ouput}

\item{genome}{Seqinfo: Genome information. If NULL the smallest common genome
will be found using bwCommonGenome when BigWig-files are analyzed.}

\item{...}{additional arguments passed to methods.}

\item{nTiles}{integer: Number of genomic tiles to parallelize over.}
}
\value{
RangedSummarizedExperiment, where assay is a sparse matrix
  (dgCMatrix) of CTSS counts and design stored in colData.
}
\description{
This function reads in CTSS count data from a series of BigWig-files (or bedGraph-files) and
returns a CTSS-by-library count matrix. For efficient processing, the count
matrix is stored as a sparse matrix (dgCMatrix from the Matrix package), and CTSSs are compressed to a GPos object if possible.
}
\examples{
\dontrun{
# Load the example data
data('exampleDesign')
# Use the BigWig-files included with the package:
bw_plus <- system.file('extdata', exampleDesign$BigWigPlus,
                       package = 'CAGEfightR')
bw_minus <- system.file('extdata', exampleDesign$BigWigMinus,
                        package = 'CAGEfightR')

# Create two named BigWigFileList-objects:
bw_plus <- BigWigFileList(bw_plus)
bw_minus <- BigWigFileList(bw_minus)
names(bw_plus) <- exampleDesign$Name
names(bw_minus) <- exampleDesign$Name

# Quantify CTSSs, by default this will use the smallest common genome:
CTSSs <- quantifyCTSSs(plusStrand=bw_plus,
                       minusStrand=bw_minus,
                       design=exampleDesign)

# Alternatively, a genome can be specified:
si <- seqinfo(bw_plus[[1]])
si <- si['chr18']
CTSSs_subset <- quantifyCTSSs(plusStrand=bw_plus,
                       minusStrand=bw_minus,
                       design=exampleDesign,
                       genome=si)

# Quantification can be speed up by using multiple cores:
library(BiocParallel)
register(MulticoreParam(workers=3))
CTSSs_subset <- quantifyCTSSs(plusStrand=bw_plus,
                       minusStrand=bw_minus,
                       design=exampleDesign,
                       genome=si)

# CAGEfightR also support bedGraph files, first BigWig is converted
bg_plus <- replicate(n=length(bw_plus), tempfile(fileext="_plus.bedGraph"))
bg_minus <- replicate(n=length(bw_minus), tempfile(fileext="_minus.bedGraph"))
names(bg_plus) <- names(bw_plus)
names(bg_minus) <- names(bw_minus)

convertBigWig2BedGraph(input=sapply(bw_plus, resource), output=bg_plus)
convertBigWig2BedGraph(input=sapply(bw_minus, resource), output=bg_minus)

# Then analyze: Note a genome MUST be supplied here!
si <- bwCommonGenome(bw_plus, bw_minus)
CTSSs_via_bg <- quantifyCTSSs(plusStrand=bg_plus,
                        minusStrand=bg_minus,
                        design=exampleDesign,
                        genome=si)

# Confirm that the two approaches yield the same results
all(assay(CTSSs_via_bg) == assay(CTSSs))
}
}
\seealso{
Other Quantification functions: 
\code{\link{quantifyCTSSs2}()},
\code{\link{quantifyClusters}()},
\code{\link{quantifyGenes}()}
}
\concept{Quantification functions}
