## ----echo = FALSE-------------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  warning = FALSE,
  message = FALSE,
  echo = TRUE,
  comment = "#>"
)

## ----echo=FALSE---------------------------------------------------------------
library(RFLOMICS)

## ----createRlomisMAE----------------------------------------------------------
data(ecoseed.mae)

factorInfo <- data.frame(
  "factorName"   = c("Repeat", "temperature", "imbibition"),
  "factorType"   = c("batch", "Bio", "Bio"),
  "factorRef"    = c("rep1", "Low", "DS")
)

# create rflomicsMAE object with ecoseed data
MAE <- createRflomicsMAE(
  projectName = "Tests",
  omicsData   =  ecoseed.mae,
  omicsTypes  = c("RNAseq", "proteomics", "metabolomics"),
  factorInfo  = factorInfo
)

names(metadata(MAE))

head(getDesignMat(MAE))

## ----designMetadata-----------------------------------------------------------
form <- generateModelFormulae(MAE)
form


MAE <- setModelFormula(MAE, form[2])


getModelFormula(MAE)

## ----contr--------------------------------------------------------------------
possibleContrasts <- generateExpressionContrast(MAE)
possibleContrasts$averaged$contrastName

MAE <- setSelectedContrasts(MAE, contrastList = possibleContrasts$averaged[1:3])

getSelectedContrasts(MAE)[, -c(1, 3)]

## ----runNorm------------------------------------------------------------------
MAE <- runDataProcessing(
  object = MAE,
  SE.name = "RNAtest",
  samples = colnames(MAE[["RNAtest"]])[-1],
  filterStrategy = "NbReplicates",
  cpmCutoff = 1,
  normMethod = "TMM"
) |>
  runDataProcessing(
    SE.name = "metatest", normMethod = "median",
    transformMethod = "log2"
  ) |>
  runDataProcessing(
    SE.name = "protetest", normMethod = "none",
    transformMethod = "log2"
  ) |>
  runOmicsPCA(SE.name = "metatest")

# Access to the normalization settings for metabolomics data

getNormSettings(object = MAE[["metatest"]])

# Obtain the list of filtered features for the RNAseq data

getFilteredFeatures(object = MAE[["RNAtest"]])[1:10]

## ----plotPCA, fig.width=10, fig.height=5, fig.show = "hold"-------------------
plotOmicsPCA(MAE[["metatest"]], raw = TRUE)
plotOmicsPCA(MAE[["metatest"]], raw = FALSE)

## ----plotDataDistribution, fig.width=10, fig.height=5, fig.show = "hold"------
plotDataDistribution(MAE[["metatest"]], raw = TRUE)
plotDataDistribution(MAE[["metatest"]], raw = FALSE)

## ----diff---------------------------------------------------------------------
MAE <- runDiffAnalysis(MAE,
  SE.name = "RNAtest", p.adj.method = "BH",
  method = "edgeRglmfit", p.adj.cutoff = 0.05, logFC.cutoff = 0
) |>
  runDiffAnalysis(
    SE.name = "protetest", p.adj.method = "BH",
    method = "limmalmFit", p.adj.cutoff = 0.05, logFC.cutoff = 0
  ) |>
  runDiffAnalysis(
    SE.name = "metatest", p.adj.method = "BH",
    method = "limmalmFit", p.adj.cutoff = 0.05, logFC.cutoff = 0
  )

# access to the settings

getDiffSettings(MAE, SE.name = "RNAtest")

# Summary of the differential analysis

getDiffStat(MAE[["RNAtest"]])

## ----volcano, fig.width=7, fig.height = 4.5, fig.align="center"---------------
plotDiffAnalysis(MAE,
  SE.name = "RNAtest",
  contrastName = "(temperatureMedium - temperatureLow) in mean",
  typeofplots = "volcano"
)

## ----boxplot, fig.width=7, fig.align="center"---------------------------------
plotBoxplotDE(MAE[["RNAtest"]],
  feature = "AT4G04810"
)

## ----coexp--------------------------------------------------------------------
MAE <- runCoExpression(MAE,
  SE.name = "protetest",
  K = 2:5, replicates = 5,
  merge = "union"
)

getCoExpAnalysesSummary(MAE)

plotCoExpression(MAE[["protetest"]])$ICL

plotCoExpressionProfile(MAE[["protetest"]], cluster = 2)

## ----enrich, fig.width=12, fig.height=5---------------------------------------
MAE <- runAnnotationEnrichment(MAE,
  SE.name = "protetest",
  OrgDb = "org.At.tair.db",
  keyType = "TAIR",
  pvalueCutoff = 0.05,
  from = "DiffExp",
  database = "GO",
  domain = "CC"
)

results <- getEnrichRes(MAE[["protetest"]],
  from = "DiffExp", database = "GO"
)

sumORA(MAE[["protetest"]], from = "DiffExp", database = "GO")

plotEnrichComp(MAE[["protetest"]],
  from = "DiffExp", database = "GO", domain = "CC",
  matrixType = "FC"
)

## -----------------------------------------------------------------------------
listIntegration <- prepareForIntegration(MAE,
  omicsNames = c("protetest", "metatest"),
  variableLists = rownames(MAE),
  method = "mixOmics"
)

MAE <- runOmicsIntegration(
  object = MAE,
  preparedObject = listIntegration,
  method = "mixOmics",
  selectedResponse = "temperature",
  ncomp = 3
)

## ----fig.width = 10, fig.align = "center"-------------------------------------
mixOmics::plotIndiv(getMixOmics(MAE, response = "temperature"))

## -----------------------------------------------------------------------------
sessionInfo()

